/**
 * @typedef {'none'|'expand'|'rotate'|'shrink'} ShiftAnimationState
 */

let canvas = null;
let ctx = null;
let width = 0;
let height = 0;
const dataUpdateRateMs = 75;
let currentRpmPercentage = 0;
let targetRpmPercentage = 0;
let rpmPercentageChange = 0;
let maxRpmValue = 0;
let currentThrottlePercentage = 0;
let targetThrottlePercentage = 0;
let throttlePercentageChange = 0;
let currentBrakePercentage = 0;
let targetBrakePercentage = 0;
let brakePercentageChange = 0;
let currentNrgPercentage = 0;
let targetNrgPercentage = 0;
let nrgPercentageChange = 0;
let speed = 0;
let gear = null;
let previousGear = null;
let defaultShiftArcLineWidth = 6;
let currentShiftArcLineWidth = defaultShiftArcLineWidth;
let maxShiftArcLineWidth = 9;
let defaultShiftArcRadius = 35;
let currentShiftArcRadius = defaultShiftArcRadius;
let minShiftArcRadius = 28;
let maxShiftArcRadius = 43;
let defaultShiftArcStartAngle = 230 / 180 * Math.PI;
let maxShiftArcStartAngle = defaultShiftArcStartAngle + 2 * Math.PI;
let currentShiftArcStartAngle = defaultShiftArcStartAngle;
const reverseRotateShiftArcStartAngle = defaultShiftArcStartAngle - 2 * Math.PI;
let defaultShiftArcEndAngle = 310 / 180 * Math.PI;
let maxShiftArcEndAngle = defaultShiftArcEndAngle + 2 * Math.PI;
let currentShiftArcEndAngle = defaultShiftArcEndAngle;
let reverseRotateShiftArcEndAngle = defaultShiftArcEndAngle - 2 * Math.PI;
const font = "bold 20px Heebo";
const throttleColor = "#81fb96";
const brakeColor = "#fd2e2f";
var f = new FontFace("Heebo", "url(assets/fonts/Heebo/Heebo.ttf)");
var f = new FontFace("Antonio", "url(assets/fonts/Antonio/Antonio.ttf)");

/** @type {ShiftAnimationState} */
let upshiftAnimationState = 'none';

/** @type {ShiftAnimationState} */
let downshiftAnimationState = 'none';

function setRpm(targetPercentage, maxRpm) {
  currentRpmPercentage = targetPercentage;
  rpmPercentageChange = Math.abs(currentRpmPercentage - targetRpmPercentage);
  maxRpmValue = maxRpm;
}

function setThrottle(targetPercentage) {
  targetThrottlePercentage = targetPercentage;
  throttlePercentageChange = Math.abs(currentThrottlePercentage - targetThrottlePercentage);
}

function setBrake(targetPercentage) {
  targetBrakePercentage = targetPercentage;
  brakePercentageChange = Math.abs(currentBrakePercentage - targetBrakePercentage);
}

function setNrg(targetPercentage) {
  targetNrgPercentage = targetPercentage;
  nrgPercentageChange = Math.abs(currentNrgPercentage - targetNrgPercentage);
}

function setSpeed(newSpeed) {
  speed = newSpeed;
}

function setGear(newGear) {
  gear = newGear;
}

function getCanvas() {
  return canvas;
}

function resetCanvas() {
  if (canvas && ctx) {
    ctx.clearRect(0, 0, width, height);
    canvas = null;
  }
}

function initCanvas(canvasElement) {
  if (!canvasElement) {
    return;
  }

  canvas = canvasElement;
  ctx = canvas.getContext("2d");

  if (!ctx) {
    canvas = null;
    return;
  }

  width = canvas.width;
  height = canvas.height;
  const startAndEndAngleInDegrees = 35;
  const outerArcRadius = height / 2;
  const hudSidePadding = height / 10;

  const totalAngleSpan = startAndEndAngleInDegrees * 2 / 180 * Math.PI;
  const endAngle = startAndEndAngleInDegrees / 180 * Math.PI;
  const startAngle = -endAngle;
  const arcY = height / 2;
  let previousTimeMs = Date.now();

  const drawHud = () => {
    ctx.clearRect(0, 0, width, height);

    const drawRpmDisplay = () => {
      const rpmArcWidth = 70;
      const rpmArcGap = 6;
      const rpmEndLineLength = 120;

      const rpmOuterArcX = outerArcRadius + hudSidePadding;
      const rpmArcRadius = outerArcRadius - rpmArcWidth / 2 - rpmArcGap;
      const rpmInnerArcRadius = outerArcRadius - rpmEndLineLength;
      const rpmArcStartAngle = startAngle + Math.PI;
      const rpmDisplayEndAngle = endAngle + Math.PI;
      const rpmArcSpan = (currentRpmPercentage / 100) * totalAngleSpan;
      const rpmArcEndAngle = rpmArcStartAngle + rpmArcSpan;

      const drawRpmOuterArc = () => {
        ctx.beginPath();
        ctx.arc(rpmOuterArcX, arcY, outerArcRadius, rpmArcStartAngle, rpmDisplayEndAngle);
        ctx.strokeStyle = "rgba(255, 255, 255, 0.8)";
        ctx.lineWidth = 2;
        ctx.stroke();
      }

      const drawRpmInnerArc = () => {
        ctx.beginPath();
        ctx.arc(rpmOuterArcX, arcY, outerArcRadius - rpmEndLineLength, rpmArcStartAngle, rpmDisplayEndAngle);
        const x0 = outerArcRadius * Math.cos(rpmArcStartAngle) + rpmOuterArcX;
        const y0 = outerArcRadius * Math.sin(rpmArcStartAngle) + arcY;
        const x1 = rpmInnerArcRadius * Math.cos(rpmDisplayEndAngle) + rpmOuterArcX;
        const y1 = rpmInnerArcRadius * Math.sin(rpmDisplayEndAngle) + arcY;
        const gradient = ctx.createLinearGradient(x0, y0, x1, y1);
        gradient.addColorStop(0, "rgba(255, 255, 255, 0.5)");
        gradient.addColorStop(0.15, "rgba(255, 255, 255, 0.5)");
        gradient.addColorStop(0.3, "transparent");
        gradient.addColorStop(0.8, "transparent");
        gradient.addColorStop(1, "rgba(255, 255, 255, 0.5)");
        ctx.strokeStyle = gradient;
        ctx.lineWidth = 2;
        ctx.stroke();
      }

      const drawRpmArcEndLine = (angle) => {
        const lineX0 = outerArcRadius * Math.cos(angle) + rpmOuterArcX;
        const lineY0 = outerArcRadius * Math.sin(angle) + arcY;
        const lineX1 = rpmInnerArcRadius * Math.cos(angle) + rpmOuterArcX;
        const lineY1 = rpmInnerArcRadius * Math.sin(angle) + arcY;
        const gradient = ctx.createLinearGradient(lineX0, lineY0, lineX1, lineY1);
        gradient.addColorStop(0, "rgba(255, 255, 255, 0.9)");
        gradient.addColorStop(1, "rgba(255, 255, 255, 0.5)");
        ctx.beginPath();
        ctx.moveTo(lineX0, lineY0);
        ctx.lineTo(lineX1, lineY1);
        ctx.strokeStyle = gradient;
        ctx.lineWidth = 2;
        ctx.stroke();
      }

      const drawRpmGuideLinesAndText = () => {
        const textOffset = 25;

        const numLines = Math.floor(maxRpmValue / 1000);
        const angleSpan = numLines * 1000 / maxRpmValue * totalAngleSpan;
        const gapAngle = angleSpan / numLines;

        for (let i = 0; i <= numLines; i++) {
          if (i < numLines) {
            const lineAngle = rpmArcStartAngle + (i + 1) * gapAngle;
            const lineX0 = outerArcRadius * Math.cos(lineAngle) + rpmOuterArcX;
            const lineY0 = outerArcRadius * Math.sin(lineAngle) + arcY;
            const lineX1 = rpmInnerArcRadius * Math.cos(lineAngle) + rpmOuterArcX;
            const lineY1 = rpmInnerArcRadius * Math.sin(lineAngle) + arcY;
            ctx.beginPath();
            ctx.moveTo(lineX0, lineY0);
            ctx.lineTo(lineX1, lineY1);
            const gradient = ctx.createLinearGradient(lineX0, lineY0, lineX1, lineY1);
            gradient.addColorStop(0, "rgba(255, 255, 255, 0.9)");
            gradient.addColorStop(0.8, "rgba(255, 255, 255, 0)");
            ctx.strokeStyle = gradient;
            ctx.lineWidth = 2;
            ctx.stroke();
          }

          ctx.font = "bold 16px Heebo";
          ctx.fillStyle = "#fff";
          ctx.textAlign = "center";
          ctx.textBaseline = "middle";
          const lineAngle = rpmArcStartAngle + i * gapAngle;
          const textX = (outerArcRadius + textOffset) * Math.cos(lineAngle) + rpmOuterArcX;
          const textY = (outerArcRadius + textOffset) * Math.sin(lineAngle) + arcY;
          ctx.save();
          ctx.translate(textX, textY);
          ctx.fillText(i.toString(), 0, 0);
          ctx.restore();
        }
      }

      const drawRpmArc = () => {
        ctx.beginPath();
        ctx.shadowColor = "rgba(200, 200, 200, 0.9)";
        ctx.shadowBlur = 20;
        ctx.arc(rpmOuterArcX, arcY, rpmArcRadius, rpmArcStartAngle, rpmArcEndAngle);
        const x0 = rpmArcRadius * Math.cos(rpmArcStartAngle) + rpmOuterArcX;
        const y0 = rpmArcRadius * Math.sin(rpmArcStartAngle) + arcY;
        const x1 = rpmArcRadius * Math.cos(rpmDisplayEndAngle) + rpmOuterArcX;
        const y1 = rpmArcRadius * Math.sin(rpmDisplayEndAngle) + arcY;
        const gradient = ctx.createLinearGradient(x0, y0, x1, y1);
        gradient.addColorStop(0, "rgba(0, 0, 50, 0.05)");
        gradient.addColorStop(0.2, "rgba(0, 0, 50, 0.3)");
        gradient.addColorStop(0.3, "rgba(0, 110, 255, 0.5)");
        gradient.addColorStop(0.55, "rgba(100, 255, 255, 0.9)");
        gradient.addColorStop(0.85, "rgba(255, 255, 255, 1)");
        ctx.strokeStyle = gradient;
        ctx.lineWidth = rpmArcWidth;
        ctx.stroke();
        ctx.shadowBlur = 0;
        ctx.shadowColor = "transparent";
      }

      drawRpmOuterArc();
      drawRpmInnerArc();
      drawRpmArcEndLine(rpmArcStartAngle);
      drawRpmArcEndLine(rpmDisplayEndAngle);
      drawRpmGuideLinesAndText();
      drawRpmArc();
      drawBgLineDisplay(rpmOuterArcX + 200, 'left')
    }

    const drawSpeedDisplay = () => {
      const x = outerArcRadius * Math.cos(180) + outerArcRadius + hudSidePadding;
      const unitOffsetY = 55;
      ctx.textAlign = "right";
      ctx.textBaseline = "middle";

      ctx.font = "bold 75px Antonio";
      ctx.fillText(speed.toFixed(), x, arcY);
      ctx.font = "24px Heebo";
      ctx.fillText('km/h', x, arcY + unitOffsetY);
    }

    const drawGearDisplay = () => {
      const gearArcRadius = 35;
      const offsetX = 135;
      const x = outerArcRadius * Math.cos(180) + outerArcRadius + hudSidePadding + offsetX;

      const drawGearArc = () => {
        ctx.beginPath();
        ctx.arc(x, arcY, gearArcRadius, 320 / 180 * Math.PI, 220 / 180 * Math.PI);
        ctx.strokeStyle = "rgba(255, 255, 255, 1)";
        ctx.lineWidth = 2;
        ctx.stroke();
      }

      const drawShiftArc = () => {
        ctx.shadowColor = "rgba(100, 255, 255, 0.9)";
        ctx.shadowBlur = 15;
        const getStrokeStyle = () => {
          switch (true) {
            case upshiftAnimationState !== 'none':
              return throttleColor;
            case downshiftAnimationState !== 'none':
              return brakeColor;
            default:
              return "rgba(255, 255, 255, 1)";
          }
        }

        ctx.beginPath();
        ctx.arc(x, arcY, currentShiftArcRadius, currentShiftArcStartAngle, currentShiftArcEndAngle);
        ctx.strokeStyle = getStrokeStyle();
        ctx.lineWidth = currentShiftArcLineWidth;
        ctx.stroke();
        ctx.shadowColor = "transparent";
        ctx.shadowBlur = 0;
      }

      const drawGearText = () => {
        const labelOffsetY = 15;
        const offsetY = 5;

        const formatGear = (gear) => {
          switch (gear) {
            case -1:
              return 'R';
            case 0:
              return 'N';
            default:
              return gear === null ? '' : gear.toString();
          }
        }

        ctx.textAlign = "center";
        ctx.textBaseline = "middle";
        ctx.font = "bold 35px Heebo";
        ctx.fillText(formatGear(gear), x, arcY + offsetY);
        ctx.font = " 20px Heebo";
        ctx.textBaseline = "top";
        ctx.fillText('GEAR', x, arcY + gearArcRadius + labelOffsetY);
      }

      drawGearArc();
      drawShiftArc();
      drawGearText();
    }

    const drawPedalDisplays = () => {
      const pedalEndLineLength = 120;
      const pedalOuterArcGap = 5;
      const pedalInnerArcGap = 8;
      const pedalArcWidth = 28;

      const pedalInnerRadius = outerArcRadius - pedalEndLineLength;
      const pedalOuterArcX = width - outerArcRadius - hudSidePadding;
      const throttleArcRadius = outerArcRadius - pedalArcWidth / 2 - pedalOuterArcGap;
      const brakeArcRadius = throttleArcRadius - pedalArcWidth - pedalInnerArcGap;
      const nrgArcRadius = brakeArcRadius - pedalArcWidth - pedalInnerArcGap;
      const labelRadius = nrgArcRadius;

      const drawPedalOuterArc = () => {
        ctx.beginPath();
        ctx.arc(pedalOuterArcX, arcY, outerArcRadius, startAngle, endAngle);
        ctx.strokeStyle = "rgba(255, 255, 255, 0.8)";
        ctx.lineWidth = 2;
        ctx.stroke();
      }

      const drawArcEndLine = (angle) => {
        const lineX0 = outerArcRadius * Math.cos(angle) + pedalOuterArcX;
        const lineY0 = outerArcRadius * Math.sin(angle) + arcY;
        const lineX1 = pedalInnerRadius * Math.cos(angle) + pedalOuterArcX;
        const lineY1 = pedalInnerRadius * Math.sin(angle) + arcY;
        const gradient = ctx.createLinearGradient(lineX0, lineY0, lineX1, lineY1);
        gradient.addColorStop(0, "rgba(255, 255, 255, 0.9)");
        gradient.addColorStop(1, "transparent");
        ctx.beginPath();
        ctx.moveTo(lineX0, lineY0);
        ctx.lineTo(lineX1, lineY1);
        ctx.strokeStyle = gradient;
        ctx.stroke();
      }

      const drawThrottleBackgroundArc = () => {
        ctx.beginPath();
        ctx.shadowColor = "rgba(129, 251, 150, 0.7)";
        ctx.arc(pedalOuterArcX, arcY, throttleArcRadius, startAngle, endAngle);
        ctx.strokeStyle = "rgba(129, 251, 150, 0.4)";
        ctx.lineWidth = pedalArcWidth;
        ctx.stroke();
      }

      const drawThrottleArc = () => {
        const throttleArcSpan = (currentThrottlePercentage / 100) * totalAngleSpan;
        const throttleArcStartAngle = endAngle - throttleArcSpan;
        ctx.beginPath();
        ctx.shadowColor = throttleColor;
        ctx.shadowBlur = 20;
        ctx.arc(pedalOuterArcX, arcY, throttleArcRadius, throttleArcStartAngle, endAngle);
        ctx.strokeStyle = throttleColor;
        ctx.lineWidth = pedalArcWidth;
        ctx.stroke();
        ctx.shadowBlur = 0;
        ctx.shadowColor = "transparent";
      }

      const drawBrakeBackgroundArc = () => {
        ctx.beginPath();
        ctx.arc(pedalOuterArcX, arcY, brakeArcRadius, startAngle, endAngle);
        ctx.strokeStyle = "rgba(253, 46, 47, 0.4)";
        ctx.lineWidth = pedalArcWidth;
        ctx.stroke();
      }

      const drawBrakeArc = () => {
        const brakeArcSpan = (currentBrakePercentage / 100) * totalAngleSpan;
        const brakeArcStartAngle = endAngle - brakeArcSpan;
        ctx.beginPath();
        ctx.shadowColor = brakeColor;
        ctx.shadowBlur = 20;
        ctx.arc(pedalOuterArcX, arcY, brakeArcRadius, brakeArcStartAngle, endAngle);
        ctx.strokeStyle = brakeColor;
        ctx.lineWidth = pedalArcWidth;
        ctx.stroke();
        ctx.shadowBlur = 0;
        ctx.shadowColor = "transparent";
      }

      const drawNrgBars = () => {
        const numRectangles = 30;
        const rectWidth = 9.5;
        const rectHeight = 21;
        const gapAngle = totalAngleSpan / numRectangles;
        const filledRectangles = Math.round((currentNrgPercentage / 100) * numRectangles);

        for (let i = 0; i < numRectangles; i++) {
          const rectAngle = endAngle - i * gapAngle - gapAngle / 2;
          const rectX = nrgArcRadius * Math.cos(rectAngle) + pedalOuterArcX;
          const rectY = nrgArcRadius * Math.sin(rectAngle) + arcY;

          ctx.save();
          ctx.translate(rectX, rectY);
          ctx.rotate(rectAngle + Math.PI / 2);

          if (i > 8) {
            ctx.fillStyle = i < filledRectangles ? "rgba(1, 253, 29, 1)" : "rgba(1, 253, 29, 0.4)";
          } else if (i > 2) {
            ctx.fillStyle = i < filledRectangles ? "rgba(252, 203, 29, 1)" : "rgba(252, 203, 29, 0.4)";
          } else {
            ctx.fillStyle = i < filledRectangles ? "rgba(253, 46, 50, 1)" : "rgba(253, 46, 50, 0.4)";
          }

          ctx.fillRect(-rectWidth / 2, -rectHeight / 2, rectWidth, rectHeight);
          ctx.shadowColor = "transparent";

          ctx.restore();
        }
      }

      const drawLabels = () => {
        ctx.font = "bold 18px Heebo";
        ctx.fillStyle = "#fff";
        const labelAngleOffset = 0.055;

        const labelEndAngle = endAngle + labelAngleOffset;
        const labelRotation = Math.PI * 1.5 + endAngle;

        const throttleLabelX = throttleArcRadius * Math.cos(labelEndAngle + 0.018) + pedalOuterArcX;
        const throttleLabelY = throttleArcRadius * Math.sin(labelEndAngle) + arcY;
        ctx.save();
        ctx.translate(throttleLabelX, throttleLabelY);
        ctx.rotate(labelRotation);
        ctx.fillText("THR", 0, 0);
        ctx.restore();

        const brakeLabelX = brakeArcRadius * Math.cos(labelEndAngle + 0.018) + pedalOuterArcX;
        const brakeLabelY = brakeArcRadius * Math.sin(labelEndAngle) + arcY;
        ctx.save();
        ctx.translate(brakeLabelX, brakeLabelY);
        ctx.rotate(labelRotation);
        ctx.fillText("BRK", 0, 0);
        ctx.restore();

        if (targetNrgPercentage > -1) {
          const nrgLabelX = labelRadius * Math.cos(labelEndAngle + 0.018) + pedalOuterArcX;
          const nrgLabelY = labelRadius * Math.sin(labelEndAngle) + arcY;
          ctx.save();
          ctx.translate(nrgLabelX, nrgLabelY);
          ctx.rotate(labelRotation);
          ctx.fillText("NRG", 0, 0);
          ctx.restore();
        }
      }

      drawPedalOuterArc();
      drawArcEndLine(startAngle);
      drawArcEndLine(endAngle);
      drawThrottleBackgroundArc();
      drawThrottleArc();
      drawBrakeBackgroundArc();
      drawBrakeArc();

      if (targetNrgPercentage > -1) {
        drawNrgBars();
      }

      drawLabels();
      drawBgLineDisplay(pedalOuterArcX - 100, 'right');
    }

    drawRpmDisplay();
    drawSpeedDisplay();
    drawGearDisplay();
    drawPedalDisplays();
  };

  const drawBgLineDisplay = (offsetX, side) => {
    const numLines = 21;
    const angleDegrees = 20;
    const radius = 500;

    const totalAngleSpan = (angleDegrees / 180) * Math.PI * 2;
    const gapAngle = totalAngleSpan / numLines;
    const endAngle = (angleDegrees / 180) * Math.PI;

    for (let i = 0; i < numLines; i++) {
      const lineAngle = endAngle - i * gapAngle - gapAngle / 2;
      const lineX0 = side === 'left'
        ? radius * Math.cos(lineAngle) * -1 + offsetX
        : radius * Math.cos(lineAngle) + offsetX;
      const lineY0 = radius * Math.sin(lineAngle) + arcY;
      let lineLength = 10;

      if (i === 0 || i === numLines - 1) {
        lineLength = 70;
      } else if (i === Math.floor(numLines / 2)) {
        lineLength = 50;
      }

      ctx.beginPath();
      ctx.moveTo(lineX0, lineY0);
      ctx.lineTo(side === 'left' ? lineX0 + lineLength : lineX0 - lineLength, lineY0);
      ctx.strokeStyle = "rgba(255, 255, 255, 0.7)";
      ctx.lineWidth = 2;
      ctx.stroke();
    }
  }

  const animateHud = () => {
    if (!canvas) {
      return;
    }

    const currentTimeMs = Date.now();
    const deltaTimeMs = currentTimeMs - previousTimeMs;
    previousTimeMs = currentTimeMs;

    const getInputCurrentPercentage = (currentPercentage, targetPercentage, percentageChange) => {
      const stepCount = dataUpdateRateMs / deltaTimeMs;
      const stepSize = percentageChange / stepCount;

      if (currentPercentage < targetPercentage) {
        currentPercentage += stepSize;

        if (currentPercentage > targetPercentage) {
          currentPercentage = targetPercentage;
        }
      } else if (currentPercentage > targetPercentage) {
        currentPercentage -= stepSize;

        if (currentPercentage < targetPercentage) {
          currentPercentage = targetPercentage;
        }
      }

      return currentPercentage;
    }

    currentThrottlePercentage = getInputCurrentPercentage(currentThrottlePercentage, targetThrottlePercentage, throttlePercentageChange);
    currentBrakePercentage = getInputCurrentPercentage(currentBrakePercentage, targetBrakePercentage, brakePercentageChange);
    currentNrgPercentage = getInputCurrentPercentage(currentNrgPercentage, targetNrgPercentage, nrgPercentageChange);

    if (gear !== previousGear) {
      if (gear > 0 && previousGear > 0) {
        if (gear > previousGear) {
          resetShiftArc();
          upshiftAnimationState = 'expand';
        }

        if (gear < previousGear) {
          resetShiftArc();
          downshiftAnimationState = 'shrink';
        }
      }

      if (gear > 0) {
        previousGear = gear;
      }
    }

    if (upshiftAnimationState === 'expand') {
      const animationLengthMs = 150;
      const widthStepSize = (maxShiftArcLineWidth - defaultShiftArcLineWidth) / animationLengthMs * deltaTimeMs;
      const radiusStepSize = (maxShiftArcRadius - defaultShiftArcRadius) / animationLengthMs * deltaTimeMs;
      currentShiftArcLineWidth += widthStepSize;
      currentShiftArcRadius += radiusStepSize;

      if (currentShiftArcLineWidth >= maxShiftArcLineWidth || currentShiftArcRadius >= maxShiftArcRadius) {
        currentShiftArcLineWidth = maxShiftArcLineWidth;
        currentShiftArcRadius = maxShiftArcRadius;
        upshiftAnimationState = 'rotate';
      }
    } else if (upshiftAnimationState === 'rotate') {
      const animationLengthMs = 500;
      const angleStepSize = 2 * Math.PI / animationLengthMs * deltaTimeMs;
      currentShiftArcStartAngle += angleStepSize;
      currentShiftArcEndAngle += angleStepSize;

      if (currentShiftArcStartAngle >= maxShiftArcStartAngle || currentShiftArcEndAngle >= maxShiftArcEndAngle) {
        currentShiftArcStartAngle = maxShiftArcStartAngle;
        currentShiftArcEndAngle = maxShiftArcEndAngle;
        upshiftAnimationState = 'shrink';
      }
    } else if (upshiftAnimationState === 'shrink') {
      const animationLengthMs = 150;
      const widthStepSize = (maxShiftArcLineWidth - defaultShiftArcLineWidth) / animationLengthMs * deltaTimeMs;
      const radiusStepSize = (maxShiftArcRadius - defaultShiftArcRadius) / animationLengthMs * deltaTimeMs;
      currentShiftArcLineWidth -= widthStepSize;
      currentShiftArcRadius -= radiusStepSize;

      if (currentShiftArcLineWidth <= defaultShiftArcLineWidth || currentShiftArcRadius <= defaultShiftArcRadius) {
        resetShiftArc();
      }
    }

    if (downshiftAnimationState === 'shrink') {
      const animationLengthMs = 100;
      const radiusStepSize = (defaultShiftArcRadius - minShiftArcRadius) / animationLengthMs * deltaTimeMs;
      currentShiftArcRadius -= radiusStepSize;

      if (currentShiftArcRadius <= minShiftArcRadius) {
        currentShiftArcRadius = minShiftArcRadius;
        downshiftAnimationState = 'rotate';
      }
    } else if (downshiftAnimationState === 'rotate') {
      const animationLength = 400;
      const angleStepSize = 2 * Math.PI / animationLength * deltaTimeMs;
      currentShiftArcStartAngle -= angleStepSize;
      currentShiftArcEndAngle -= angleStepSize;

      if (currentShiftArcStartAngle <= reverseRotateShiftArcStartAngle || currentShiftArcEndAngle <= reverseRotateShiftArcEndAngle) {
        currentShiftArcStartAngle = reverseRotateShiftArcStartAngle;
        currentShiftArcEndAngle = reverseRotateShiftArcEndAngle;
        downshiftAnimationState = 'expand';
      }
    } else if (downshiftAnimationState === 'expand') {
      const animationLengthMs = 100;
      const radiusStepSize = (defaultShiftArcRadius - minShiftArcRadius) / animationLengthMs * deltaTimeMs;
      currentShiftArcRadius += radiusStepSize;

      if (currentShiftArcRadius >= defaultShiftArcRadius) {
        resetShiftArc();
      }
    }

    drawHud();
    requestAnimationFrame(animateHud);
  };

  function resetShiftArc() {
    currentShiftArcLineWidth = defaultShiftArcLineWidth;
    currentShiftArcRadius = defaultShiftArcRadius;
    currentShiftArcStartAngle = defaultShiftArcStartAngle;
    currentShiftArcEndAngle = defaultShiftArcEndAngle;
    upshiftAnimationState = 'none';
    downshiftAnimationState = 'none';
  }

  animateHud();
}
