angular.module('overlay.battleBox', []).component('battleBoxOverlay', {
    bindings: {
        overlayInfo: '<',
        selectedDriver: '<',
        config: '<',
        customConfig: '<',
        standings: '<',
        speedUnit: '<',
        entryDataField: '<',
        selectedCarClass: '<',
        selectedEntryInfo: '<',
        cars: '<'
    },
    templateUrl: 'src/components/battleboxoverlay/battlebox.overlay.html',
    controller: function (broadcastService, sessionService, standingsService, utilService, $http, $scope) {
        var ctrl = this;
        ctrl.leadingCarImageSrc = null;
        ctrl.trailingCarImageSrc = null;
        $scope.leadingDriver = null;
        $scope.trailingDriver = null;

        $scope.$watch('leadingDriver', function(newVal, oldVal) {
            if (newVal && oldVal && newVal.slotID !== oldVal.slotID || (newVal && !oldVal)) {
                broadcastService.getDiskOrGameCarImageSrc(
                    newVal,
                    _.get(ctrl, 'config.customConfig'),
                    setLeadingCarImageSrc
                );
            }
        });

        $scope.$watch('trailingDriver', function(newVal, oldVal) {
            if (newVal && oldVal && newVal.slotID !== oldVal.slotID || (newVal && !oldVal)) {
                broadcastService.getDiskOrGameCarImageSrc(
                    newVal,
                    _.get(ctrl, 'config.customConfig'),
                    setTrailingCarImageSrc
                );
            }
        });

        function setLeadingCarImageSrc(src) {
            ctrl.leadingCarImageSrc = src;
        }

        function setTrailingCarImageSrc(src) {
            ctrl.trailingCarImageSrc = src;
        }

        this.$onChanges = function(changes) {
            if (changes.selectedDriver) {
                setLeadingAndTrailingDriver(changes.selectedDriver.currentValue, ctrl.standings);
            }
        }

        function setLeadingAndTrailingDriver(selectedDriver, standings) {
            if (!selectedDriver || !standings) {
                return;
            }

            var driverBehind;
            var driverAhead;
            if (broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                driverBehind = _.find(standings, { position: selectedDriver.position + 1 });
                driverAhead = _.find(standings, { position: selectedDriver.position - 1 });
            } else {
                driverBehind = _.find(
                    standings,
                    { classPosition: selectedDriver.classPosition + 1, carClass: selectedDriver.carClass }
                );

                driverAhead = _.find(
                    standings,
                    { classPosition: selectedDriver.classPosition - 1, carClass: selectedDriver.carClass }
                )
            }

            if (!driverAhead) {
                // Selected driver is P1
                $scope.leadingDriver = selectedDriver;
                $scope.trailingDriver = driverBehind;
            } else {
                if (!driverBehind) {
                    // Selected driver is last
                    $scope.leadingDriver = driverAhead ? driverAhead : selectedDriver;
                    $scope.trailingDriver = selectedDriver;
                } else {
                    if (hasCloserBattleWithCarInFront(selectedDriver, driverBehind)) {
                        $scope.leadingDriver = driverAhead ? driverAhead : selectedDriver;
                        $scope.trailingDriver = selectedDriver;
                    } else {
                        $scope.leadingDriver = selectedDriver;
                        $scope.trailingDriver = driverBehind;
                    }
                }
            }
        }

        // Returns true if selected driver has a closer battle with car in front than with the car behind
        function hasCloserBattleWithCarInFront(selectedDriver, driverBehind) {
            if (broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                return Math.abs(selectedDriver.timeBehindNext) < Math.abs(driverBehind.timeBehindNext)
                    || driverBehind.lapsBehindNext > 0;
            }

            return selectedDriver.timeBehindNextInClass < driverBehind.timeBehindNextInClass
                || driverBehind.lapsBehindNextInClass > 0;
        }

        this.getCssCarClassName = function(carClass) {
            return utilService.generateCssCarClassName(carClass);
        }

        this.getBorderLeftStyle = function(driver) {
            return broadcastService.getDriverBorderStyle(driver, ctrl.customConfig, 'borderLeft');
        }
        this.getStandingsGroupBorderStyle = function(carClassName) {
            var classColor = broadcastService.getCarClassColorFromConfig(ctrl.customConfig, carClassName);

            if (!classColor) {
                return null;
            }

            return {
                borderLeft: 'solid 0.2em ' + classColor
            }
        }

        this.getStandingsGroupRightBorderStyle = function(carClassName) {
            var classColor = broadcastService.getCarClassColorFromConfig(ctrl.customConfig, carClassName);

            if (!classColor) {
                return null;
            }

            return {
                borderRight: 'solid 0.2em ' + classColor
            }
        }

        this.getSwipeStyle = function() {
            return broadcastService.getCarClassColorStyle(
                ctrl.selectedDriver,
                ctrl.customConfig,
                'backgroundColor'
            );
        }

        this.getOptionalDataSetting = function() {
            return _.get(ctrl, 'overlayInfo.settings.optionalData');
        }

        // Value to be shown in the compare box on the left (for the leading driver)
        this.getLeftCompareValue = function(leadingDriver, trailingDriver) {
            if (!ctrl.overlayInfo)  {
                return '';
            }

            var optionalDataSetting = ctrl.getOptionalDataSetting();

            if (optionalDataSetting === 'lastLapTime' || optionalDataSetting === 'bestLapTime') {
                return standingsService.formatDriverLapTime(leadingDriver, optionalDataSetting);
            } else if (optionalDataSetting === 'velocity') {
                var carSpeedValue = _.get(leadingDriver, 'carVelocity.velocity');
                var speedUnit = _.get(ctrl, 'speedUnit');

                return standingsService.convertCarSpeed(carSpeedValue, speedUnit).toFixed(0);
            }

            return '-'; // Hidden with CSS
        }

        this.leftCompareBoxValueIsHidden = function(leadingDriver) {
            return _.get(ctrl, 'overlayInfo.settings.optionalData') === 'default';
        }

        // Text to be shown in the compare box on the left (for the leading driver)
        this.getLeftCompareText = function(leadingDriver, trailingDriver) {
            if (!ctrl.overlayInfo) {
                return;
            }

            if (ctrl.getOptionalDataSetting() === 'lastLapTime') {
                return 'Last lap time';
            } else if (ctrl.getOptionalDataSetting() === 'bestLapTime') {
                return 'Best lap time';
            } else if (ctrl.getOptionalDataSetting() === 'velocity') {
                return ctrl.speedUnit;
            }

            return 'Gap';
        }

        // Value to be shown in the compare box on the right (for the trailing driver)
        this.getCompareRightValue = function(leadingDriver, trailingDriver) {
            var optionalDataSetting = ctrl.getOptionalDataSetting();

            if (optionalDataSetting === 'lastLapTime' || optionalDataSetting === 'bestLapTime') {
                // Display lap time difference
                var lapTimeDiff = standingsService.getLapTimeDiff(leadingDriver, trailingDriver, optionalDataSetting);

                return standingsService.formatLapTimeDiff(lapTimeDiff);
            } else if (optionalDataSetting === 'velocity') {
                return Math.abs(getSpeedDifference(leadingDriver, trailingDriver));
            }

            // Default to displaying live gap
            return standingsService.displayLiveGapToNextDriver(
                trailingDriver,
                !broadcastService.isMixedClassMode(ctrl.selectedCarClass)
            );
        }

        // Text to be shown in the compare box on the right (for the trailing driver)
        this.getCompareRightText = function(leadingDriver, trailingDriver) {
            var optionalDataSetting = ctrl.getOptionalDataSetting();

            if (optionalDataSetting === 'lastLapTime' || optionalDataSetting === 'bestLapTime') {
                var lapTimeDiff = standingsService.getLapTimeDiff(leadingDriver, trailingDriver, optionalDataSetting);

                if (lapTimeDiff === null) {
                    return '-';
                }

                if (lapTimeDiff > 0) {
                    return 'Slower';
                }

                return 'Faster';
            } else if (optionalDataSetting === 'velocity') {
                var speedDiff = getSpeedDifference(leadingDriver, trailingDriver);

                if (speedDiff > 0) {
                    return 'Slower';
                }

                if (speedDiff < 0) {
                    return 'Faster';
                }

                return '-';
            }

            // Default to gap
            return 'Behind';
        }

        function getSpeedDifference(leadingCar, trailingCar) {
            var leadingCarSpeedValue = _.get(leadingCar, 'carVelocity.velocity');
            var trailingCarSpeedValue = _.get(trailingCar, 'carVelocity.velocity');
            var speedUnit = _.get(ctrl, 'speedUnit');
            var leadingDisplaySpeed = standingsService
                .convertCarSpeed(leadingCarSpeedValue, speedUnit).toFixed(0);
            var trailingDisplaySpeed = standingsService
                .convertCarSpeed(trailingCarSpeedValue, speedUnit).toFixed(0);

            return leadingDisplaySpeed - trailingDisplaySpeed;
        }
    }
});
