angular.module('controlpanel.cameraControl', [
    'BroadcastService',
    'SessionService',
    'StandingsService',
    'UtilService',
    'VehicleInfoService'
]).component('cameraControl', {
    bindings: {
        standings: '<',
        selectedDriver: '<',
        selectDriver: '&',
        selectCamera: '&',
        sessionInfo: '<',
        removeColumns: '<?',
        customConfig: '<',
        customConfigName: '<?',
        carClasses: '<',
        currentCarClass: '<', // Used for displaying timing gaps etc.
        standingsHistory: '<',
        dynamicVehicleClassColors: '<',
        tableConfig: '<?',
        units: '<?', // 0 = metric, 1 = imperial
        detailedPitStateMap: '<?',
        filteredClasses: '<'
    },
    templateUrl: 'src/components/cameracontrol/cameracontrol.html',
    controller: function (broadcastService, sessionService, standingsService, utilService, vehicleInfoService, $interval) {
        this.selectedCarClassFilter = null; // Used for filtering the table.
        let scrollStartAt;
        let scrollInterval;
        const defaultStickyRowCount = 3;
        const defaultMaxRowCount = 20;
        const defaultScrollIntervalMs = 10000;
        this.tableStandings = [];

        this.$onChanges = ((changes) => {
            if (changes.filteredClasses) {
                console.log('filteredClasses updated:', changes.filteredClasses.currentValue);
            }
            if (changes.standings) {
                if (this.tableConfig?.liveTimingPage) {
                    if (!this.customConfig) {
                        return;
                    }

                    if (this.isLiveTimingMulticlassMode()) {
                        this.tableStandings = groupStandingsByCarClass(changes.standings.currentValue);
                    } else {
                        const stickyRowCount = this.customConfig?.liveTiming?.stickyRows ?? defaultStickyRowCount;
                        const tableMaxRowCount = this.customConfig?.liveTiming?.tableMaxRows ?? defaultMaxRowCount;

                        if (scrollStartAt == null) {
                            scrollStartAt = stickyRowCount;
                        }

                        const stickyEntries = changes.standings.currentValue.slice(0, stickyRowCount);
                        const scrollingEntries = changes.standings.currentValue.slice(
                            scrollStartAt,
                            scrollStartAt + (tableMaxRowCount - stickyRowCount)
                        );

                        this.tableStandings = stickyEntries.concat(scrollingEntries);

                        if (needScrolling() && !scrollInterval) {
                            const delay = this.customConfig?.liveTiming?.cycleRowsDelaySeconds * 1000 || defaultScrollIntervalMs;

                            scrollInterval = $interval(() => {
                                scrollStartAt = scrollStartAt + (tableMaxRowCount - stickyRowCount);

                                if (scrollStartAt > this.standings.length) {
                                    scrollStartAt = stickyRowCount;
                                }
                            }, delay);
                        }
                    }
                } else {
                    this.tableStandings = changes.standings.currentValue;
                }
            }
        });

        const needScrolling = () => {
            return this.standings?.length > (this.customConfig?.liveTiming?.tableMaxRows ?? defaultMaxRowCount);
        }

        const groupStandingsByCarClass = (standings) => {
            const grouped = _.groupBy(standings, 'carClass');
            const multiclassRowCount = this.customConfig?.liveTiming?.multiClassRowCount;
            let carClassGroups;

            if (multiclassRowCount > 0) {
                carClassGroups = Object.values(grouped).map(carClassGroup => {
                    const sliceEnd = this.selectedCarClassFilter == null ? multiclassRowCount : carClassGroup.length;

                    return carClassGroup.slice(0, sliceEnd);
                });
            } else {
                carClassGroups = Object.values(grouped).flat();
            }

            return carClassGroups.flat();
        }

        this.isLiveTimingMulticlassMode = () => {
            return this.tableConfig?.liveTimingPage && this.customConfig?.liveTiming?.multiClassMode;
        }

        this.getLatestDriverData = function(driver) {
            var driverLatest = _.find(this.standings, { slotID: driver.slotID });

            if (driverLatest) {
                return driverLatest;
            }

            return driver;
        }

        this.getPosition = (driver) => {
            if (this.isLiveTimingMulticlassMode()) {
                return driver.classPosition;
            } else {
                return this.selectedCarClassFilter ? driver.classPosition : driver.position;
            }
        }

        this.getClassPosition = (driver) => {
            return driver.classPosition;
        }

        this.getCarNumber = function(driver) {
            return vehicleInfoService.getCarNumberOverride(driver);
        }

        this.getDriverName = function(driver) {
            return vehicleInfoService.getDriverNameOverride(driver);
        }

        this.getTeamName = (driver) => {
            return vehicleInfoService.getTeamNameOverride(driver);
        }

        this.getCarImageSrc = (driver) => {
            return broadcastService.getCarImageSrc(driver, this.customConfigName);
        }

        this.getCarFallbackImageSrc = (driver) => {
            return broadcastService.getCarGameImageSrc(driver.vehicleFilename);
        }

        this.getBattleIndicatorClassAttr = function(driver) {
            if (!sessionService.isRaceSession(this.sessionInfo)) {
                return;
            }

            var classAtrr = '';
            let timeBehind;
            let timeAhead;

            if (broadcastService.isMixedClassMode(this.currentCarClass)) {
                timeBehind = Math.abs(_.get(driver, 'timeBehindNext'));
                const driverBehind = _.find(this.standings, { position: driver.position + 1 });
                timeAhead = driverBehind ? Math.abs(driverBehind.timeBehindNext) : null;
            } else {
                timeBehind = Math.abs(_.get(driver, 'timeBehindNextInClass'));
                timeAhead = Math.abs(_.get(driver, 'timeAheadPreviousInClass'));
            }

            if ((timeBehind > 0 && timeBehind < 0.5) || (timeAhead > 0 && timeAhead < 0.5)) {
                classAtrr = 'red';
            } else if ((timeBehind > 0 && timeBehind < 1) || (timeAhead > 0 && timeAhead < 1)) {
                classAtrr = 'white';
            } else if ((timeBehind > 0 && timeBehind < 2) || (timeAhead > 0 && timeAhead < 2)) {
                classAtrr = 'grey';
            }

            return classAtrr;
        }

        this.formatLapTime = function(lapTime) {
            return utilService.secToString(lapTime, 3);
        }

        this.displayBestLapTime = function(driver) {
            return this.formatLapTime(this.getLatestDriverData(driver).bestLapTime);
        }

        this.displayLastLapTime = function(driver) {
            return this.formatLapTime(this.getLatestDriverData(driver).lastLapTime);
        }

        this.displayGapCellValue = function(driver) {
            if (sessionService.isRaceSession(this.sessionInfo)) {
                return standingsService.displayLiveGapToLeader(driver, !broadcastService.isMixedClassMode(this.currentCarClass));
            }

            if ((broadcastService.isMixedClassMode(this.currentCarClass) && driver.position === 1)
                || (!broadcastService.isMixedClassMode(this.currentCarClass) && driver.classPosition === 1)) {
                return '';
            }

            var leader = standingsService.findLeader(this.standings, broadcastService.isMixedClassMode(this.currentCarClass) ? null : driver.carClass);
            var lapTimeDiff = standingsService.getLapTimeDiff(leader, driver, 'bestLapTime');

            return standingsService.formatLapTimeDiff(lapTimeDiff);
        }

        this.displayIntervalCellValue = function(driver) {
            if (sessionService.isRaceSession(this.sessionInfo)) {
                return standingsService.displayLiveGapToNextDriver(driver, !broadcastService.isMixedClassMode(this.currentCarClass));
            }

            if ((broadcastService.isMixedClassMode(this.currentCarClass) && driver.position === 1)
                || (!broadcastService.isMixedClassMode(this.currentCarClass) && driver.classPosition === 1)) {
                return '';
            }

            var nextDriver = standingsService.findNextDriver(
                this.standings,
                driver,
                broadcastService.isMixedClassMode(this.currentCarClass) ? null : driver.carClass
            );
            var lapTimeDiff = standingsService.getLapTimeDiff(nextDriver, driver, 'bestLapTime');

            return standingsService.formatLapTimeDiff(lapTimeDiff);
        }

        this.displaySectorTimeCellValue = function(driver, sector) {
            driver = this.getLatestDriverData(driver);

            var sectorTime;

            if (sessionService.isRaceSession()) {
                sectorTime = getCurrentOrLastSectorTime(driver, sector);
            } else {
                if (driver.pitting) {
                    sectorTime = standingsService.getDriverBestLapSectorTime(driver, sector);
                } else {
                    sectorTime = getCurrentOrLastSectorTime(driver, sector);
                }
            }

            if (sectorTime === undefined) {
                return '';
            } else if (sectorTime <= 0) {
                return '-';
            }

            return sectorTime.toFixed(3);
        }

        function getCurrentOrLastSectorTime(driver, sector) {
            var sectorTime;

            switch(sector) {
                case 1:
                    if (driver.currentSectorTime1 !== -1) {
                        sectorTime = driver.currentSectorTime1;
                    }

                    break;
                case 2:
                    if (driver.currentSectorTime1 === -1 && driver.currentSectorTime2 === -1) {
                        // Driver currently in sector 1
                        sectorTime = driver.lastSectorTime2 - driver.lastSectorTime1;
                    } else if (driver.currentSectorTime1 !== -1 && driver.currentSectorTime2 !== -1) {
                        // Driver currently in sector 3
                        sectorTime = driver.currentSectorTime2 - driver.currentSectorTime1;
                    }

                    break;
                case 3:
                    if (driver.lastLapTime !== -1
                        && (driver.currentSectorTime1 === -1 || driver.currentSectorTime2 === -1)) {
                        // Driver currently in either sector 1 or sector 2
                        sectorTime = driver.lastLapTime - driver.lastSectorTime2;
                    }
            }

            return sectorTime;
        }

        this.getSectorCellClass = function(driver, sector) {
            driver = this.getLatestDriverData(driver);

            var cellClass = {};

            if ((sector === 1 || sector === 2)
                && driver['bestSectorTime' + sector] === -1 && driver.bestLapTime === -1) {
                return;
            }

            if (sector === 3 && driver.bestLapTime === -1) {
                return;
            }

            if (this.getGreenSector(driver, sector)) {
                cellClass['green-time'] = true;
            }

            if (this.getPurpleSector(driver, sector)) {
                cellClass['purple-time'] = true;
            }

            return cellClass;
        }

        this.getGreenSector = function(driver, sector) {
            return broadcastService.displayGreenSector(this.getLatestDriverData(driver), sector, this.sessionInfo);
        }

        this.getPurpleSector = function(driver, sector) {
            let purpleSectorTime;
            let displayPurpleSectorTime;
            let isCorrectCarClass;

            if (broadcastService.isMixedClassMode(this.currentCarClass)) {
                purpleSectorTime = _.get(this, 'sessionInfo.lapTimeInfo.purpleSectors.' + sector + '.time');
                isCorrectCarClass = true;
            } else {
                purpleSectorTime = broadcastService.getPurpleSectorTime(
                    this.getLatestDriverData(driver).carClass,
                    sector,
                    this.sessionInfo
                );

                isCorrectCarClass = driver.carClass === this.currentCarClass
                    || broadcastService.isMulticlassClassMode(this.currentCarClass);
            }

            displayPurpleSectorTime = broadcastService.displayPurpleSector(
                this.getLatestDriverData(driver),
                sector,
                sessionService.isRaceSession(this.sessionInfo),
                purpleSectorTime
            );

            return displayPurpleSectorTime && isCorrectCarClass;
        }

        this.isGreenLastLapTime = function(driver) {
            driver = this.getLatestDriverData(driver);

            if (driver.lastLapTime === -1) {
                return false;
            }

            return driver.lastLapTime === driver.bestLapTime;
        }

        this.isPurpleLapTime = function(driver, lapType) {
            driver = this.getLatestDriverData(driver);

            if (driver[lapType + 'LapTime'] === -1) {
                return false;
            }

            var sessionBestLapTime = standingsService.findSessionBestLapTime(
                this.standings,
                broadcastService.isMixedClassMode(this.currentCarClass) ? null : driver.carClass
            );

            return driver[lapType + 'LapTime'] === sessionBestLapTime
                && (driver.carClass === this.currentCarClass
                    || broadcastService.isMixedClassMode(this.currentCarClass)
                    || broadcastService.isMulticlassClassMode(this.currentCarClass)
                );
        }

        /**
         * @param {Driver} driver
         * @returns {string}
         */
        this.getDriverStatus = function(driver) {
            const detailedPitState = this.detailedPitStateMap.get(driver.slotID)?.detailedPitState ?? null;

            return standingsService.formatDriverStatus(this.getLatestDriverData(driver), detailedPitState);
        }

        /**
         * @param {Driver} driver
         * @returns {boolean}
         */
        this.isPitStatus = (driver) => {
            const driverStatus = this.getDriverStatus(driver);

            return driverStatus === 'Pit'
              || driverStatus === 'Pit in'
              || driverStatus === 'Box'
              || driverStatus === 'Pit out';
        }

        this.getCarSpeed = (driver) => {
          const velocityMs = driver.carVelocity?.velocity ?? 0;

          return (this.units === 1 ? velocityMs * 2.236936 : velocityMs * 3.6).toFixed();
        }

        this.formatCarClassName = (carClass) => {
            return utilService.generateCssCarClassName(carClass);
        }

        this.getRowBgColor = function(entry, rowIndex) {
            let rowBgColor;
            let canUseCarClassBgColor = false;

            if (this.tableConfig?.liveTimingPage) {
                if (this.isLiveTimingMulticlassMode()) {
                    canUseCarClassBgColor = true;
                }
            } else {
                if (this.selectedCarClassFilter == null) {
                    canUseCarClassBgColor = true;
                }
            }

            if (canUseCarClassBgColor) {
                const configCarClassBgColor = _.get(
                    this, 'customConfig.colors.carClasses.' + entry.carClass + '.backgroundColor'
                );

                if (configCarClassBgColor) {
                    let { h, s, l } = utilService.hexToHSL(configCarClassBgColor);

                    if (h == null || s == null || l == null) {
                        rowBgColor = this.getRowDefaultBgColor(rowIndex);
                    } else {
                        l = rowIndex % 2 === 0 ? '58' : '45';
                        rowBgColor = `hsl(${h},${s}%,${l}%, 0.5)`;
                    }
                } else {
                    rowBgColor = this.getRowDefaultBgColor(rowIndex);
                }
            } else {
                rowBgColor = this.getRowDefaultBgColor(rowIndex);
            }

            return rowBgColor;
        }

        this.getRowDefaultBgColor = function(rowIndex) {
            // Base bg color on row being odd or even.
            return rowIndex % 2 === 0 ?  '#1a1a1a' : '#262626';
        }

        this.onClickCarClass = function(carClass) {
            if (this.tableConfig?.liveTimingPage && !this.customConfig?.liveTiming?.multiClassMode) {
                return;
            }

            if (carClass === this.selectedCarClassFilter) {
                this.selectedCarClassFilter = null;
            } else {
                this.selectedCarClassFilter = carClass;
            }
        }

        this.removeColumn = function(columnName) {
            return _.includes(this.removeColumns, columnName);
        }

        this.getTireColor = (axle, driver) => {
            const tireConfigColor = getTireConfigProp('color', axle, driver);

            return tireConfigColor || '#000';
        }

        this.getTireCompoundName = (axle, driver) => {
            let name = '';
            const tireConfigName = getTireConfigProp('name', axle, driver);

            if (tireConfigName) {
                name = tireConfigName.substring(0, 2);
            } else {
                const tireIngameName = standingsService.getTireCompoundName(driver, axle);

                if (tireIngameName) {
                    name = tireIngameName.substring(0, 1);
                }
            }

            return name;
        }

        const getTireConfigProp = (propName, axle, driver) => {
            const tireName = standingsService.getTireCompoundName(driver, axle);
            const carClass = driver?.carClass;

            return _.get(this, `customConfig.tires.${carClass}.${tireName}.${propName}`);
        }
    }
}).filter('filterByCarClass', function() {
    return function(standings, selectedCarClassFilter) {
        if (!selectedCarClassFilter) {
            return standings;
        }

        return _.filter(standings, { carClass: selectedCarClassFilter });
    }
});
