angular.module('overlay.standingsTicker', []).component('standingsTickerOverlay', {
    bindings: {
        overlayInfo: '<',
        sessionInfo: '<',
        drivers: '<',
        carClass: '<',
        customConfig: '<',
        pitTimers: '<',
        pitLaneTimerEnabled: '<',
        pitBoxTimerEnabled: '<',
        entryDataField: '<',
        startingOrder: '<',
        selectedNameType: '<',
        driverNameTypeId: '<',
        cars: '<'
    },
    templateUrl: 'src/components/standingstickeroverlay/standingsticker.overlay.html',
    controller: function (
        broadcastService,
        sessionService,
        standingsService,
        utilService,
        $animate,
        $timeout,
        $scope
    ) {
        var ctrl = this;
        ctrl.scrollTimeout = null;
        ctrl.scrollLeft = false;
        ctrl.moveRight = false;
        ctrl.limitIndex = 0;
        ctrl.elementLimit = 4; // Number of elements to show at a time
        ctrl.scrollingTimeoutDelay = 10000; // Set default, set later from value in custom config if available

        this.$onChanges = function(changes) {
            if (changes.drivers) {
                var currentStandings = changes.drivers.currentValue;
                var previousStandings = changes.drivers.previousValue;

                if (ctrl.carClass && !broadcastService.isMixedClassMode(ctrl.carClass)) {
                    // Non-mixed mode, sort or filter the drivers based on the mode.
                    if (broadcastService.isMulticlassClassMode(ctrl.carClass)) {
                        // Multi-class mode
                        var currentCarClassRanking = getCarClassRanking(currentStandings);
                        var previousCarClassRanking = getCarClassRanking(previousStandings);
                        currentStandings = sortStandingsInMulticlassMode(
                            changes.drivers.currentValue,
                            currentCarClassRanking
                        );
                        previousStandings = sortStandingsInMulticlassMode(
                            changes.drivers.previousValue,
                            previousCarClassRanking
                        );
                    } else {
                        // Individual car class mode
                        currentStandings = _.filter(currentStandings, { carClass: ctrl.carClass });
                        previousStandings = _.filter(previousStandings, { carClass: ctrl.carClass });
                    }
                }


                if (currentStandings.length <= ctrl.elementLimit) {
                    // No need for scrolling the banner. Just show the banner slots if not yet doing so.
                    if (ctrl.scrollTimeout) {
                        cancelScrolling();
                    }

                    ctrl.limitIndex = 0;
                    if (!ctrl.showSlots) {
                        $timeout(function() {
                            ctrl.showSlots = true;
                        }, 1000);
                    }
                } else {
                    // Start scrolling if not already doing so.
                    if (ctrl.scrollTimeout === null) {
                        $timeout(function() {
                            animateScroll(ctrl.scrollingTimeoutDelay);
                        }, 1000);
                    }
                }

                if (currentStandings && previousStandings && currentStandings.length !== previousStandings.length) {
                    return;
                }

                animatePositionChange(currentStandings, previousStandings);
                ctrl.drivers = currentStandings;
            }

            if (changes.customConfig) {
                var scrollingInterval = _.get(ctrl, 'customConfig.overlays.banner.scrollingInterval');
                if (scrollingInterval > 0) {
                    ctrl.scrollingTimeoutDelay = scrollingInterval;
                }
            }

            if (changes.carClass) {
                var currentCarClass = _.get(changes, 'carClass.currentValue');
                var previousCarClass = _.get(changes, 'carClass.previousValue');
                if (currentCarClass && previousCarClass && currentCarClass !== previousCarClass) {
                    // Selected car class changed, reset banner cycling animation and start it again
                    cancelScrolling();
                    ctrl.showSlots = false;
                    ctrl.limitIndex = 0;
                    $timeout(function() {
                        animateScroll(ctrl.scrollingTimeoutDelay);
                    }, 1000);
                }
            }
        }

        function getCarClassRanking(standings) {
            var carClassRanking = {};
            _.forEach(standings, function(entry) {
                if (!carClassRanking[entry.carClass] || entry.position < carClassRanking[entry.carClass]) {
                    carClassRanking[entry.carClass] = entry.position;
                }
            });

            return carClassRanking;
        }

        // Returns sorted standings/drivers for multiclass mode
        function sortStandingsInMulticlassMode(standings, carClassRanking) {
            if (standings && standings.length > 0) {
                return standings.sort(function(a, b) {
                    if (a.carClass === b.carClass) {
                        // Same car class, sort by overall position
                        return a.position - b.position;
                    }

                    // Different car class, sort by car class ranking
                    return carClassRanking[a.carClass] - carClassRanking[b.carClass];
                });
            }
        }

        function cancelScrolling() {
            $timeout.cancel(ctrl.scrollTimeout);
            ctrl.scrollTimeout = null;
        }

        // Scrolls the list when needed
        function animateScroll(scrollingTimeoutDelay) {
            ctrl.showSlots = true;

            if (ctrl.scrollTimeout) {
                cancelScrolling();
            }

            if (!ctrl.drivers || ctrl.drivers.length <= ctrl.elementLimit) {
                return;
            }


            ctrl.scrollTimeout = $timeout(function() {
                if (ctrl.drivers.length <= ctrl.elementLimit) {
                    return;
                }

                ctrl.showSlots = false;

                $timeout(function() {
                    ctrl.limitIndex += ctrl.elementLimit;
                    if (ctrl.limitIndex > ctrl.drivers.length - 1) {
                        ctrl.limitIndex = 0;
                    }

                    $timeout(function() {
                        animateScroll(scrollingTimeoutDelay);
                    }, 50);
                }, 500);
            }, scrollingTimeoutDelay);
        }

        $scope.$on('$destroy', function() {
            cancelScrolling();
        });

        this.getBorderLeftStyle = function(driver) {
            var classColor = broadcastService.getCarClassColorFromConfig(ctrl.customConfig, driver.carClass);

            if (!classColor) {
                return null;
            }

            return {
                borderLeft: 'solid 0.2em ' + classColor
            }
        }

        this.getEntryPitDuration = function(entry, propName) {
            var pitStopEntry = _.find(ctrl.pitTimers, { slotID: entry.slotID });

            if (!pitStopEntry) {
                return null;
            }

            return pitStopEntry[propName];
        }

        this.getFadedOut = function(driver) {
            return standingsService.getEntryFadedOut(driver, ctrl.sessionInfo);
        }

        this.getCssCarClassName = function(carClass) {
            return utilService.generateCssCarClassName(carClass);
        }

        this.displayTime = function(entry) {
            return standingsService.displayTime(entry, this.sessionInfo);
        }

        function animatePositionChange(currentStandings, oldVal) {
            _.forEach(currentStandings, function(newEntry, indexOfNewEntry) {
                var indexOfOldEntry = _.findIndex(oldVal, function(oldEntry) {
                    return newEntry.slotID === oldEntry.slotID;
                });

                if (indexOfOldEntry > -1 && indexOfOldEntry !== indexOfNewEntry) {
                    var elem = document.getElementsByClassName('ticker-slot-id-' + newEntry.slotID)[0];
                    if (!elem) {
                        return;
                    }

                    var moveAmount;
                    if (indexOfNewEntry < indexOfOldEntry) {
                        moveAmount = (indexOfOldEntry - indexOfNewEntry) * 100;
                    } else {
                        moveAmount = ((indexOfNewEntry - indexOfOldEntry) * 100) * -1;
                    }

                    elem.style.transform = 'translateX(' + moveAmount + '%)';
                    $animate.addClass(elem, 'animate-move').then(() => {
                        elem.classList.remove('animate-move');
                    });

                    $timeout(() => {
                        elem.style.transform = 'translateX(0)';
                    }, 0);
                }
            });
        }
    }
}).directive('entryAnimationWrapper', function() {
    return {
        link: function($scope, $elem) {
            function broadcastAnimEvent() {
                $scope.$broadcast('CONDENSE_TEXT:SCALE_TEXT');
            }

            $elem[0].addEventListener('animationend', broadcastAnimEvent);
            $scope.$on('$destroy', function() {
                $elem[0].removeEventListener('animationend', broadcastAnimEvent);
            });
        }
    }
});
