var overlaysApp = angular.module('overlaysApp', [
    'ngAnimate',
    'BroadcastService',
    'SessionService',
    'StandingsService',
    'UtilService',
    'VehicleInfoService',
    'overlay.driverInfo',
    'overlay.battleBox',
    'overlay.broll',
    'overlay.countdown',
    'overlay.eventSlide',
    'overlay.extendedBattleBox',
    'overlay.inraceSeasonStandings',
    'overlay.introSlide',
    'overlay.onboardDriverBox',
    'overlay.sessionResults',
    'overlay.sessionInfo',
    'overlay.qualifyingBox',
    'overlay.raceUpdate',
    'overlay.replayTransition',
    'overlay.scheduleSlide',
    'overlay.seasonStandings',
    'overlay.standingsTicker',
    'overlay.standingsTower',
    'overlay.startingOrder',
    'overlay.trackMap',
    'overlay.videoSlide',
    'condenseText',
    'dynamicDataField',
    'entryDataField',
    'entryInfo',
    'entryNameField',
    'focusIndicator',
    'partnerImages',
    'positionField',
    'resultHeader',
    'gridHeader',
    'scheduleList',
    'onSrcError'
]);

overlaysApp.run(function($http) {
    $http.defaults.headers.post['Content-Type'] = 'text/plain';
});

overlaysApp.controller('OverlaysController', function(
  broadcastService,
  sessionService,
  standingsService,
  utilService,
  vehicleInfoService,
  $scope,
  $http,
  $interval,
  $timeout
) {
    var restUrl = '/rest/watch/';
    var webdataUrl = '/webdata/';
    var ctrl = this;

    /**
     * @type {SessionInfo | null}
     */
    ctrl.sessionInfo = null;
    ctrl.sessionType; // Session type in lowercase.
    ctrl.drivers = []; // Driver list used in tower, banner etc.
    ctrl.updateDriversTimeout = null;
    ctrl.standings = [];
    ctrl.focusDriver = null; // Current driver in focus from the back-end.
    ctrl.selectedDriver = null;  // Updated as often as possible from standings based on the focusDriver.
    ctrl.overlays = {};
    ctrl.sessionResults = [];
    ctrl.midraceResults = [];
    ctrl.startingOrder = [];
    ctrl.trackMap = null;
    ctrl.config = null;
    ctrl.customConfig = null;
    ctrl.showReplayText = false;
    ctrl.displayOptions = null;
    ctrl.speedUnit = '';
    ctrl.lapStartTimes = [];
    ctrl.pauseOverlayUpdates = false;
    ctrl.seasonStandings = null;
    ctrl.cars = [];
    ctrl.pitTimers = [];
    ctrl.pitLaneTimerEnabled = true;
    ctrl.pitBoxTimerEnabled = true;
    ctrl.schedule = [];
    ctrl.replayMetrics = {}; // Data from replay metrics WebSocket.
    ctrl.liveStandings = []; // For now, use tire compound data from the UI LiveStandings WS.
    ctrl.canvasWidth = screen.height * 16 / 9;
    ctrl.canvasHeight = screen.height;
    var wsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/overlays';
    var replayMetricsWsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/replaymetrics';
    var ws;
    var replayMetricsWs;
    var uiWsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/ui';
    var uiWs;

    // Elements in driver box overlay. The text in these elements is updated directly from replay metrics WebSocket data
    // to skip the slower Angular binding updates.
    var carSpeedValueElement;
    var carRpmValueElement;
    var brakeBarElement;
    var throttleBarElement;

    getDisplayOptions();
    getFocusDriver();
    initWebSocket();

    $interval(function() {
        // Check that we have a WebSocket connection
        if (!ws || ws.readyState === 3) {
            initWebSocket();
        }
    }, 5000);

    // For now, use tire compound data from the UI LiveStandings WS.
    function initUiWebSocket() {
        uiWs = new WebSocket(uiWsUrl);

        uiWs.onopen = function() {
            const subMsg = {
                messageType: 'SUB',
                topic: 'LiveStandings'
            }

            uiWs.send(JSON.stringify(subMsg));

            uiWs.onmessage = function(event) {
                var message = JSON.parse(event.data);

                if (message && message.topic === 'LiveStandings') {
                    ctrl.liveStandings = message.body;
                }
            }
        }
    }

    function initWebSocket() {
        ws = new WebSocket(wsUrl);

        ws.onopen = function() {
            // Get latest values from back-end with a single GET request.
            broadcastService.getSessionInfo().then(function(response) {
                ctrl.sessionInfo = response.data;
                ctrl.sessionType = getSessionType(response.data);
            });

            broadcastService.getStandings().then(function(response) {
                ctrl.standings = response.data;
            });

            broadcastService.getStandingsHistory().then(function(response) {
                ctrl.standingsHistory = response.data;
            });
        }

        ws.onmessage = function(event) {
            var message = JSON.parse(event.data);

            if (message && message.type === 'standings') {
                updateLapStartTimes(ctrl.lapStartTimes, message.body, ctrl.standings, ctrl.sessionInfo);
                updatePitTimers(
                  ctrl.pitTimers,
                  message.body,
                  ctrl.standings,
                  ctrl.sessionInfo
                );
                updateStandings(message.body);

                if (ctrl.focusDriver) {
                    ctrl.selectedDriver = _.find(ctrl.standings, function(entry) {
                        return entry.slotID === ctrl.focusDriver.slotID;
                    });
                }

                if (ctrl.selectedDriver) {
                    var lapStartTimeEntry = _.find(ctrl.lapStartTimes, {slotID: ctrl.selectedDriver.slotID});

                    if (lapStartTimeEntry) {
                        ctrl.selectedDriver.lapStartEventTime = lapStartTimeEntry.lapStartEventTime;
                    }
                }

                // For now, use tire compound data from the UI LiveStandings WS.
                if (ctrl.liveStandings) {
                    _.forEach(ctrl.liveStandings, liveStandingsEntry => {
                        const standingsEntry = ctrl.standings.find(e => e.slotID === liveStandingsEntry.slotID);

                        standingsEntry.tireCompoundNameFront = liveStandingsEntry.tireCompoundNameFront;
                        standingsEntry.tireCompoundNameRear = liveStandingsEntry.tireCompoundNameRear;
                    });
                }

                $scope.$apply();
            }

            if (message && message.type === 'standingsHistory') {
                ctrl.standingsHistory = message.body;
                $scope.$apply();
            }

            if (message && message.type === 'sessionInfo') {
                /**
                 * @type {SessionInfo}
                 */
                ctrl.sessionInfo = message.body;
                ctrl.sessionType = getSessionType(message.body);
                var carClassGroups = broadcastService.getCarClassGroups(ctrl.standings, 'position');
                ctrl.sessionInfo.lapTimeInfo = {
                    purpleSectors: standingsService.getSessionFastestSectors(ctrl.standingsHistory),
                    purpleSectorsInClass: standingsService.getSessionFastestSectorsInClass(
                      ctrl.standingsHistory,
                      carClassGroups
                    )
                }

                $scope.$apply();
            }
        }
    }

    function getSessionType(sessionInfo) {
        return sessionService.getSessionType(_.get(sessionInfo, 'session')).toLowerCase()
    }

    function connectReplayMetricsWebSocket() {
        replayMetricsWs = new WebSocket(replayMetricsWsUrl);

        replayMetricsWs.addEventListener('open', () => {
            replayMetricsWs.addEventListener('message', handleReplayMetricsMessage);
        });
    }

    function handleReplayMetricsMessage(event) {
        if (!event || !event.data) {
            return;
        }

        const jsonData = JSON.parse(event.data);
        if (jsonData && jsonData.type === 'replayMetrics') {

            if (jsonData.body) {
                ctrl.replayMetrics = jsonData.body;

                // Update driver box's speed and RPM elements innerText value directly instead of using Angular binding,
                // since that does not update quickly enough.
                updateDriverBoxInnerText(carSpeedValueElement, jsonData.body.speed);
                updateDriverBoxInnerText(carRpmValueElement, jsonData.body.rpm);

                // Update driver box's pedal inputs.
                updateDriverBoxPedalValues(brakeBarElement, '--brake-value', jsonData.body.brakes);
                updateDriverBoxPedalValues(throttleBarElement, '--throttle-value', jsonData.body.throttle);

                // Update data for canvas HUD.
                const nrgRemaining = jsonData.body.virtualEnergyRemaining;
                setThrottle((jsonData.body.throttle || 0) * 100);
                setBrake((jsonData.body.brakes || 0) * 100);
                setNrg(nrgRemaining === -1 ? -1 : (nrgRemaining || 0) * 100);
                const maxRpm = jsonData.body.maxRPM;
                setRpm(Math.min(jsonData.body.rpm / maxRpm * 100, 100), maxRpm);
                setSpeed((jsonData.body.speed ?? 0));
                setGear(jsonData.body.gear);
            }
        }
    }

    function updateDriverBoxInnerText(element, innerTextValue) {
        if (!element) {
            return;
        }

        if (innerTextValue == null) {
            element.innerText = '-';
        } else {
            element.innerText = Math.floor(innerTextValue);
        }
    }

    function updateDriverBoxPedalValues(element, customCssProperty, cssValue) {
        if (!element) {
            return;
        }

        element.style.setProperty(customCssProperty, cssValue);
    }

    broadcastService.getTrackMap().then(function(response) {
        ctrl.trackMap = response.data;
    });

    broadcastService.getAllVehicles().then(function(vehicles) {
        ctrl.cars = vehicles;
    });

    $interval(function() {
        getStartingOrder();
        getFocusDriver();
    }, 1000);

    $scope.$on('COUNTDOWN:REACHED_ZERO', function() {
        ctrl.pauseOverlayUpdates = true;

        $timeout(function() {
            ctrl.overlays.countdown.visible = false;
            broadcastService.saveOverlays(ctrl.overlays).then(function() {
                $timeout(function() {
                    ctrl.overlays.videoSlide.visible = true;
                    broadcastService.saveOverlays(ctrl.overlays);
                    ctrl.pauseOverlayUpdates = false;
                }, 1000);
            });
        }, 1000);
    });

    $scope.$on('VIDEO_SLIDE:LAST_VIDEO_ENDED', function() {
        ctrl.pauseOverlayUpdates = true;
        ctrl.overlays.videoSlide.visible = false;

        broadcastService.saveOverlays(ctrl.overlays).then(function() {
            $timeout(function() {
                ctrl.overlays.introSlide.visible = true;
                broadcastService.saveOverlays(ctrl.overlays);
                ctrl.pauseOverlayUpdates = false;
            }, 1000);
        });
    });

    broadcastService.getConfig().then(config => {
        if (config) {
            getCustomConfigJson(config.customConfig).then(function() {
                ctrl.config = config;
            }).catch(function() {
                config.customConfig = 'default';
                ctrl.config = config;

                getCustomConfigJson('default').then(function(defaultCustomConfig) {
                    ctrl.customConfig = defaultCustomConfig;
                });
            }).finally(function() {
                getOverlayData().then(function(overlayData) {
                    vehicleInfoService.getVehicleOverrideInfo(overlayData.selectedCustomOverlay).then(function(overrideInfo) {
                        vehicleInfoService.setVehicleOverrideInfo(overrideInfo);
                    }).catch(function() {
                        vehicleInfoService.setVehicleOverrideInfo(null);
                    });
                });

                $interval(function() {
                    if (!ctrl.pauseOverlayUpdates) {
                        getOverlayData();
                    }
                }, 500);
            });
        }
    });

    function getCustomConfigJson(customConfigName) {
        return broadcastService.getCustomConfig(customConfigName).then(customConfig => {
            ctrl.customConfig = customConfig;
            ctrl.customConfig.configName = customConfigName;
            var pitLaneTimerEnabled = _.get(ctrl, 'customConfig.pitStopTimers.pitLaneTimer.enabled');
            var pitBoxTimerEnabled = _.get(ctrl, 'customConfig.pitStopTimers.pitBoxTimer.enabled');
            ctrl.pitLaneTimerEnabled = _.isNil(pitLaneTimerEnabled) || pitLaneTimerEnabled === true;
            ctrl.pitBoxTimerEnabled = _.isNil(pitBoxTimerEnabled) || pitBoxTimerEnabled === true;

            return customConfig;
        });
    }

    function getResults() {
        var selectedSessionResultsSession = _.get(ctrl, 'overlays.sessionResults.settings.selectedSession');
        ctrl.selectedSessionResultsSessionName = selectedSessionResultsSession;

        if (!selectedSessionResultsSession) {
            return;
        }

        broadcastService.getSessionResults(selectedSessionResultsSession).then(function(response) {
            if (response.status === 200) {
                ctrl.sessionResults = response.data;
            }
        }).catch(function() {
            ctrl.sessionResults = [];
        });
    }

    function shouldShowCarData() {
        if (!ctrl.overlays) {
            return undefined;
        }

        const driverInfoIsVisible = !!_.get(ctrl, 'overlays.driverInfo.visible');
        const driverInfoIsShowingCarData = !!_.get(ctrl, 'overlays.driverInfo.settings.showCarData');
        const onboardHudIsVisible = !!_.get(ctrl, 'overlays.onboardHud.visible');

        return (driverInfoIsVisible && driverInfoIsShowingCarData) || onboardHudIsVisible;
    }

    function replayMetricsWebSocketIsOpenOrConnecting() {
        return replayMetricsWs && (replayMetricsWs.readyState === 0 || replayMetricsWs.readyState === 1);
    }

    function getOverlayData() {
        return broadcastService.getOverlays().then(function(response) {
            if (response.status === 200) {
                ctrl.overlays = response.data;
                if (response.data.selectedCustomOverlay) {
                    ctrl.config.customConfig = response.data.selectedCustomOverlay;
                    getCustomConfigJson(response.data.selectedCustomOverlay).then(function() {
                        if (ctrl.config && ctrl.config.customConfig) {
                            broadcastService.getSeasonStandingsJson(ctrl.config.customConfig).then(function(seasonStandings) {
                                ctrl.seasonStandings = seasonStandings;
                            }).catch(function() {
                                ctrl.seasonStandings = [];
                            });

                            broadcastService.getScheduleJson(ctrl.config.customConfig).then(function(schedule) {
                                ctrl.schedule = schedule;
                            }).catch(function() {
                                ctrl.schedule = [];
                            });
                        }
                    }).catch(function() {
                        ctrl.seasonStandings = null;
                        ctrl.schedule = null;
                    });
                } else {
                    ctrl.config.customConfig = 'default';
                }

                // Open/close replay metrics WebSocket when needed.
                if (shouldShowCarData()) {
                    if (!replayMetricsWebSocketIsOpenOrConnecting()) {
                        connectReplayMetricsWebSocket();
                    }

                    if (!carSpeedValueElement) {
                        carSpeedValueElement = document.querySelector('.car-speed-value');
                    }

                    if (!carRpmValueElement) {
                        carRpmValueElement = document.querySelector('.car-rpm-value');
                    }

                    if (!brakeBarElement) {
                        brakeBarElement = document.querySelector('.driver-box .brake-bar');
                    }

                    if (!throttleBarElement) {
                        throttleBarElement = document.querySelector('.driver-box .throttle-bar');
                    }
                } else {
                    if (replayMetricsWebSocketIsOpenOrConnecting()) {
                        replayMetricsWs.removeEventListener('message', handleReplayMetricsMessage);
                        replayMetricsWs.close();
                        carSpeedValueElement = null;
                        carRpmValueElement = null;
                    }
                }

                if (ctrl.overlays.onboardHud.visible) {
                  if (!getCanvas()) {
                    initCanvas(document.getElementById('canvas'));
                  }
                } else {
                  resetCanvas();
                }

                if (ctrl.overlays.midraceResults.visible) {
                    broadcastService.getMidraceResults().then(function(midraceResults) {
                        ctrl.midraceResults = midraceResults;
                    }).catch(function() {
                        ctrl.midraceResults = [];
                    });
                }

                if (_.get(ctrl, 'overlays.refreshOverlaysPage.visible')) {
                    location.reload(true);
                }

                if (isShowingTireCompound()) {
                    if (!uiWs || uiWs.readyState === 3) {
                        // For now, use tire compound data from the UI LiveStandings WS.
                        initUiWebSocket();
                    }
                } else {
                    if (uiWs && uiWs.readyState === 1) {
                        uiWs.close();
                    }
                }
            }

            return response.data;
        }).catch(function(reason) {
            ctrl.overlays = broadcastService.getDefaultElementConfig(ctrl.customConfig);
            if (!ctrl.overlays.carClass) {
                ctrl.overlays.carClass = broadcastService.getOverlayStartMode(ctrl.customConfig);
            }

            return reason;
        }).finally(function() {
            getResults();
        });
    }

    function isShowingTireCompound() {
        return _.get(ctrl, 'overlays.standingsTower.dynamicData') === 'tireName'
          && (_.get(ctrl, 'overlays.standingsTower.visible') || _.get(ctrl, 'overlays.standingsTicker.visible'));
    }

    function updateStandings(standings) {
        if (!ctrl.updateDriversTimeout) {
            ctrl.updateDriversTimeout = $timeout(function() {
                ctrl.updateDriversTimeout = null;
            }, 1000);

            // Only update driver list used in tower etc. once per second
            ctrl.drivers = ctrl.standings;
        }

        var driversSorted = _.sortBy(standings, 'position');
        ctrl.standings = driversSorted;
        var carClassGroups = broadcastService.getCarClassGroups(driversSorted);
        broadcastService.addStandingsProps(driversSorted, carClassGroups);

        // Add driver lap time (and sector time) info
        if (ctrl.sessionInfo && ctrl.sessionInfo.lapTimeInfo) {
            _.forEach(standings, function(entry) {
                entry.lapTimeInfo = {};

                for (var sector = 1; sector <= 3; sector++) {
                    entry.lapTimeInfo['bestSector' + sector] = standingsService
                      .findBestSectorTime(ctrl.standingsHistory[entry.slotID], sector);

                    if (entry.lapTimeInfo['bestSector' + sector] === ctrl.sessionInfo.lapTimeInfo['purpleSectors'][sector].time) {
                        entry.lapTimeInfo['hasPurpleSector' + sector] = true;
                    }

                    var classPurpleSectorTime = _.get(
                      ctrl,
                      'sessionInfo.lapTimeInfo.purpleSectorsInClass.' + entry.carClass + '.' + sector + '.time'
                    );

                    if (classPurpleSectorTime && entry.lapTimeInfo['bestSector' + sector] === classPurpleSectorTime) {
                        entry.lapTimeInfo['hasPurpleSector' + sector + 'InClass'] = true;
                    }
                }
            });
        }
    }

    function updatePitTimers(pitTimers, currentStandings, previousStandings, sessionInfo) {
        if (!previousStandings
          || previousStandings.length === 0
          || !sessionInfo
          || !sessionService.isRaceSession(sessionInfo)
          || (!ctrl.pitLaneTimerEnabled && !ctrl.pitBoxTimerEnabled)
        ) {
            return;
        }

        _.forEach(currentStandings, function(entry) {
            var previousEntry = _.find(previousStandings, {slotID: entry.slotID});

            if (!previousEntry) {
                return; // Continue
            }

            if (standingsService.driverIsInGarageStall(entry)
              || standingsService.driverHasFinishedSession(entry)
            ) {
                cancelPitTimer(pitTimers, entry, 'pitBox');
                cancelPitTimer(pitTimers, entry, 'pitLane');
            } else if (driverEnteredPitLane(entry, previousEntry)) {
                if (ctrl.pitLaneTimerEnabled) {
                    updatePitTimerEntry(pitTimers, entry, 'pitLane', updatePitLaneTimer);
                }
            } else if (driverEnteredPitBox(entry, previousEntry)) {
                if (ctrl.pitBoxTimerEnabled) {
                    updatePitTimerEntry(pitTimers, entry, 'pitBox', updatePitBoxTimer);
                }
            } else if (driverLeftPitBox(entry, previousEntry)) {
                cancelPitTimer(pitTimers, entry, 'pitBox');
            } else if (driverLeftPitLane(entry, previousEntry)) {
                cancelPitTimer(pitTimers, entry, 'pitLane');
            }
        });
    }

    function updatePitTimerEntry(pitTimers, entry, propPart, timerUpdateFn) {
        var pitTimerEntry = _.find(pitTimers, {slotID: entry.slotID});
        if (pitTimerEntry) {
            pitTimerEntry[propPart + 'Duration'] = 0;
        } else {
            pitTimerEntry = {
                slotID: entry.slotID,
                [propPart + 'Duration']: 0
            }

            ctrl.pitTimers.push(pitTimerEntry);
        }

        pitTimerEntry[propPart + 'Counter'] = $interval(timerUpdateFn, 100, 0, true, pitTimerEntry);
    }

    function driverEnteredPitLane(currentEntry, previousEntry) {
        return standingsService.driverIsEnteringPitLane(currentEntry)
          && !standingsService.driverIsEnteringPitLane(previousEntry);
    }

    function driverEnteredPitBox(currentEntry, previousEntry) {
        return standingsService.driverIsInPitBox(currentEntry)
          && standingsService.driverIsEnteringPitLane(previousEntry);
    }

    function driverLeftPitBox(currentEntry, previousEntry) {
        return standingsService.driverIsExitingPitLane(currentEntry)
          && standingsService.driverIsInPitBox(previousEntry);
    }

    function driverLeftPitLane(currentEntry, previousEntry) {
        return !standingsService.isDriverPitting(currentEntry)
          && (standingsService.driverIsExitingPitLane(previousEntry)
            // Pit state can be NONE after a driver swap when car is driving towards the pit exit:
            || (previousEntry && previousEntry.pitState === 'NONE')
            || standingsService.driverIsEnteringPitLane(previousEntry));
    }

    function updatePitLaneTimer(pitLaneTimer) {
        pitLaneTimer.pitLaneDuration += 0.1;
    }

    function updatePitBoxTimer(pitBoxTimer) {
        pitBoxTimer.pitBoxDuration += 0.1;
    }

    function cancelPitTimer(pitTimers, entry, propPart) {
        var pitTimerIndex = _.findIndex(ctrl.pitTimers, {slotID: entry.slotID});

        if (pitTimerIndex > -1) {
            $interval.cancel(ctrl.pitTimers[pitTimerIndex][propPart + 'Counter']);
            ctrl.pitTimers[pitTimerIndex][propPart + 'Duration'] = 0;

            var otherPropPart = propPart === 'pitLane' ? 'pitBox' : 'pitLane';
            if (!ctrl.pitTimers[pitTimerIndex][otherPropPart + 'Counter']) {
                // Remove the timer entry if the other timer is not active
                ctrl.pitTimers.splice(pitTimerIndex, 1);
            }
        }
    }

    // Save the current event time for each driver in lapStartTimes array if they crossed the line.
    // To be used for displaying a lap time timer/counter in the qualifying box overlay.
    function updateLapStartTimes(lapStartTimes, currentStandings, previousStandings, sessionInfo) {
        if (!previousStandings || previousStandings.length === 0 || !sessionInfo) {
            return;
        }

        _.forEach(currentStandings, function(entry) {
            var previousEntry = _.find(previousStandings, {slotID: entry.slotID});
            var lapStartTimeEntry = _.find(lapStartTimes, {slotID: entry.slotID});
            var currentEventTime = _.get(sessionInfo, 'currentEventTime');

            if (previousEntry && entry.lapsCompleted > previousEntry.lapsCompleted) {
                if (lapStartTimeEntry) {
                    lapStartTimeEntry.lapStartEventTime = currentEventTime;
                } else {
                    ctrl.lapStartTimes.push({
                        slotID: entry.slotID,
                        lapStartEventTime: currentEventTime
                    });
                }
            } else if (entry.inGarageStall && lapStartTimeEntry) {
                // Reset lap start time when in garage stall
                lapStartTimeEntry.lapStartEventTime = null;
            }
        });
    }

    function getFocusDriver() {
        $http.get(restUrl + 'focus').then(function(focus) {
            if (focus.data !== -1) {
                ctrl.focusDriver = _.find(ctrl.standings, function(entry) {
                    return entry.slotID === focus.data;
                });
            }
        });
    }

    function getDisplayOptions() {
        broadcastService.getDisplayOptions().then(function(displayOptions) {
            ctrl.displayOptions = displayOptions;
            ctrl.speedUnit = getSpeedUnit(displayOptions.GAMEOPT_kph);
        });
    }

    function getStartingOrder() {
        broadcastService.getStartingOrder().then(function(startingOrder) {
            ctrl.startingOrder = startingOrder;
        }).catch(function() {
            ctrl.startingOrder = [];
        });
    }

    function getSpeedUnit(setting) {
        if (setting.currentValue === 0) {
            return 'mph';
        }

        return 'km/h';
    }

    ctrl.getCssCarClassName = function(carClass) {
        return utilService.generateCssCarClassName(carClass);
    }

    ctrl.getSwipeStyle = function(driver, colorProp) {
        return broadcastService.getCarClassColorStyle(driver, ctrl.customConfig, colorProp);
    }

    ctrl.getCarClassSwipeStyle = function(carClass, colorProp) {
        if (!carClass) {
            return;
        }

        return {
            [colorProp]: broadcastService.getCarClassColorFromConfig(ctrl.customConfig, carClass)
        }
    }
});

overlaysApp.filter('addSuffix', () => {
    return (number) => {
        let j = number % 10;
        let k = number % 100;
        let numberWithSuffix = number + 'th';

        if (j == 1 && k != 11) {
            numberWithSuffix = number + 'st';
        } else if (j == 2 && k != 12) {
            numberWithSuffix = number + 'nd';
        } else if (j == 3 && k != 13) {
            numberWithSuffix = number + 'rd';
        }

        return numberWithSuffix;
    }
});
