let raceResults = null;

angular.module('BroadcastService', ['VehicleInfoService']).service('broadcastService', function (
    $http,
    $q,
    sessionService,
    standingsService,
    utilService,
    vehicleInfoService
) {
    var restUrl = '/rest/watch/';
    var webdataUrl = '/webdata/';
    var optionsRestUrl = '/rest/options/';
    const sessionsRestUrl = '/rest/sessions/';
    var clients = [];

    return {
        setClients,
        getOverlayStartMode,
        getDefaultElementConfig,
        getConfig,
        getCustomConfig,
        getOverlays,
        saveOverlays,
        getCustomOverlayFolders,
        getSessionInfo,
        getStandings,
        getStandingsHistory,
        getSessionResults,
        saveSessionResults,
        getResultSessionNames,
        getMidraceResults,
        saveMidraceResults,
        getStartingOrder,
        saveStartingOrder,
        getReplayEvents,
        setReplayEvents,
        getTrackMap,
        jumpToReplayTime,
        focusOnDriver,
        getCameras,
        getActiveCamera,
        selectCamera,
        getDisplayOptions,
        getSeasonStandingsJson,
        getScheduleJson,
        getCarClassGroups,
        addStandingsProps,
        getCarClassColorFromConfig,
        getBorderStyle,
        getDriverBorderStyle,
        getCarClassColorStyle,
        getAllVehicles,
        getRaceResults,
        getCarGameImageSrc,
        getCarGameImage,
        getCarImageSrc,
        getTeamImageSrc,
        getDiskOrGameCarImageSrc,
        displayGreenSector,
        getPurpleSectorTime,
        displayPurpleSector,
        disableOtherOverlays,
        enableOtherOverlays,
        checkSessionResults,
        getCustomConfigSeriesName,
        getCustomConfigTrackName,
        getCustomConfigSeriesColor,
        getDefaultSeriesLogoUrl,
        getGameLogoUrl,
        isMixedClassMode,
        isMulticlassClassMode,
        getCurrentRoundIndex,
        getCurrentTrackNameFromSchedule,
        getAvailableDynamicVehicleClassColors,
        getDynamicVehicleClassColors,
        findVehicleClassesNeedingDynamicColor,
        getVehicleClassesInfo
    }

    function setClients(selectedClients) {
        clients = selectedClients;
    }

    function getOverlayStartMode(customConfig) {
        return _.get(customConfig, 'startInMulticlassMode') === true
            ? 'Multiclass'
            : 'Mixed'
    }

    function getDefaultElementConfig(customConfig = null) {
        return {
            selectedCustomOverlay: null,
            carClass: getOverlayStartMode(customConfig),
            entryDataField: {
                selected: 'carNumber'
            },
            sessionInfo: {
                name: 'Session Info',
                key: 'sessionInfo',
                visible: false,
                settings: {
                    code80: {
                        enabled: false
                    },
                    safetyCar: {
                        enabled: false
                    }
                },
                disableOverlays: ['midraceResults', 'sessionResults', 'startingOrder']
            },
            broll: {
                name: 'B-roll',
                key: 'broll',
                visible: false,
                settings: {
                    autoTogglingIsEnabled: false
                },
                disallowAutomaticDisabling: true
            },
            countdown: {
                name: 'Countdown',
                key: 'countdown',
                visible: false,
                disableOverlays: { group: 'all' }
            },
            driverInfo: {
                name: 'Driver box',
                key: 'driverInfo',
                visible: false,
                settings: {
                    showCarData: false,
                    entryInfo: 'driverAndTeam'
                },
                disableOverlays: ['extendedBattleBox', 'midraceResults', 'sessionResults', 'battleBox', 'qualifyingBox', 'startingOrder']
            },
            onboardDriverBox: {
                name: 'On-board driver box',
                key: 'onboardDriverBox',
                visible: false,
            },
            onboardHud: {
                name: 'On-board HUD',
                key: 'onboardHud',
                visible: false,
            },
            battleBox: {
                name: '1v1 battle',
                key: 'battleBox',
                visible: false,
                settings: {
                    optionalData: 'default'
                },
                disableOverlays: ['extendedBattleBox', 'midraceResults', 'sessionResults', 'driverInfo', 'qualifyingBox', 'startingOrder']
            },
            eventSlide: {
                name: 'Event',
                key: 'eventSlide',
                visible: false,
                settings: {
                    showCustomData: false
                },
                disableOverlays: { group: 'all', except: ['introSlide'] }
            },
            extendedBattleBox: {
                name: 'Extended battle',
                key: 'extendedBattleBox',
                visible: false,
                disableOverlays: {
                    group: 'all',
                    except: ['raceUpdateBox', 'standingsTicker', 'sessionInfo', 'standingsTower']
                }
            },
            ingameOverlay: {
                name: 'In-game overlay',
                key: 'ingameOverlay',
                visible: true,
                disallowAutomaticDisabling: true,
                disableOverlays: []
            },
            ingamePanels: {
                name: 'In-game panels',
                key: 'ingamePanels',
                visible: true,
                disallowAutomaticDisabling: true,
                disableOverlays: []
            },
            inraceSeasonStandings: {
                name: 'In-race season standings',
                key: 'inraceSeasonStandings',
                visible: false,
                settings: {
                    selectedCarClass: getOverlayStartMode(customConfig) === 'Mixed' ? 'Mixed' : null,
                    page: 1
                },
                disableOverlays: {
                    group: 'all',
                    except: ['standingsTicker', 'sessionInfo', 'standingsTower']
                }
            },
            introSlide: {
                name: 'Intro',
                key: 'introSlide',
                visible: false,
                disableOverlays: { group: 'all', except: ['eventSlide'] }
            },
            qualifyingBox: {
                name: 'Qualifying',
                key: 'qualifyingBox',
                visible: false,
                disableOverlays: ['extendedBattleBox', 'midraceResults', 'sessionResults', 'battleBox', 'driverInfo', 'startingOrder']
            },
            raceUpdateBox: {
                name: 'Race update box',
                key: 'raceUpdateBox',
                visible: false
            },
            refreshOverlaysPage: {
                name: 'Refresh overlays page',
                key: 'refreshOverlaysPage',
                visible: false,
                disallowAutomaticDisabling: true,
                disableOverlays: []
            },
            scheduleSlide: {
                name: 'Schedule',
                key: 'scheduleSlide',
                visible: false,
                settings: {
                    showingWhen: 'prerace'
                },
                disableOverlays: { group: 'all' }
            },
            seasonStandings: {
                name: 'Season standings',
                key: 'seasonStandings',
                visible: false,
                settings: {
                    selectedCarClass: getOverlayStartMode(customConfig) === 'Mixed' ? 'Mixed' : null,
                    driversPerPage: 10,
                    page: 1
                },
                disableOverlays: { group: 'all' }
            },
            standingsTower: {
                name: 'Tower',
                key: 'standingsTower',
                visible: false,
                driversPerPage: 45,
                page: 1,
                showName: 'Driver',
                dynamicData: 'gapToLeader',
                disableOverlays: ['midraceResults', 'standingsTicker', 'sessionResults', 'startingOrder'],
                carClass: getOverlayStartMode(customConfig),
                hideField: null,
                driverNameTypeId: 'lastName'
            },
            standingsTicker: {
                name: 'Banner',
                key: 'standingsTicker',
                visible: false,
                disableOverlays: ['midraceResults', 'standingsTower', 'sessionResults', 'startingOrder']
            },
            standingsInfo: {
                name: 'Standings Info',
                key: 'standingsInfo',
                visible: false,
                driversPerPage: 20,
                page: 1,
                disableOverlays: ['standingsTicker', 'sessionResults', 'startingOrder']
            },
            startingOrder: {
                name: 'Starting order',
                key: 'startingOrder',
                visible: false,
                settings: {
                    selectedCarClass: getOverlayStartMode(customConfig) === 'Mixed' ? 'Mixed' : null,
                },
                disableOverlays: {
                    group: 'all'
                }
            },
            sessionResults: {
                name: 'Session results',
                key: 'sessionResults',
                visible: false,
                settings: {
                    selectedCarClass: getOverlayStartMode(customConfig) === 'Mixed' ? 'Mixed' : null,
                    selectedSession: null,
                    driversPerPage: 10,
                    page: 1,
                },
                disableOverlays: {
                    group: 'all'
                }
            },
            midraceResults: {
                name: 'Mid-race results',
                key: 'midraceResults',
                visible: false,
                settings: {
                    selectedCarClass: getOverlayStartMode(customConfig) === 'Mixed' ? 'Mixed' : null,
                    driversPerPage: 10,
                    page: 1,
                },
                disableOverlays: {
                    group: 'all'
                }
            },
            replayTransition: {
                name: 'Show replay',
                key: 'replayTransition',
                visible: false,
                replayIsActive: false,
                activeReplayEventIndex: -1,
                settings: {
                    selectedDriver: null,
                    selectedCamera: null,
                    secondsToGoBack: 30
                },
                disableOverlays: {
                    group: 'all',
                    except: ['driverInfo', 'sessionInfo'],
                    disableInstantly: false
                },
                enableOverlays: {
                    overlays: ['driverInfo', 'sessionInfo'],
                    enableInstantly: false
                }
            },
            videoSlide: {
                name: 'Play video',
                key: 'videoSlide',
                visible: false,
                disableOverlays: { group: 'all' }
            }
        };
    }

    function getConfig() {
        return $http.get('overlays-config.json').then(function(response) {
            return response.data;
        });
    }

    function getCustomConfig(customConfigName) {
        if (!customConfigName) {
            return $q.reject();
        }

        return $http.get('customize/' + customConfigName + '/config.json').then(function(response) {
            return response.data;
        });
    }

    function getOverlays () {
        return $http.get(webdataUrl + 'overlays');
    }

    function saveOverlays(overlays) {
        var urlPart = 'overlays';
        postOverlayDataToClients(urlPart, overlays);

        return $http.post(webdataUrl + urlPart, overlays);
    }

    function getCustomOverlayFolders() {
        return $http.get('overlays-list.json').then(function(response) {
            return response.data;
        });
    }

    function postOverlayDataToClients(urlPart, data) {
        if (clients.length > 0) {
            _.forEach(clients, function(client) {
                var clientUrl = location.protocol + '//' + client.ip + ':' + client.port + webdataUrl + urlPart;

                $http.post(clientUrl, data);
            });
        }
    }

    /**
     * @returns {Promise<HttpPromiseResponse<SessionInfo>>}
     */
    function getSessionInfo () {
        return $http.get(restUrl + 'sessionInfo').then(function (response) {
            return response;
        });
    }

    function getStandings () {
        return $http.get(restUrl + 'standings').then(function (response) {
            return response;
        });
    }

    function getStandingsHistory() {
        return $http.get(restUrl + 'standings/history').then(function (response) {
            return response;
        });
    }

    function getSessionResults(sessionName) {
        return $http.get(webdataUrl + 'results/' + sessionName).catch(function (reason) {
            return reason;
        }).then(function (response) {
            return response;
        });
    }

    function saveSessionResults(sessionResults, sessionName) {
        var urlPart = 'results/' + sessionName;
        postOverlayDataToClients(urlPart, sessionResults);

        return $http.post(webdataUrl + urlPart, sessionResults);
    }

    function getResultSessionNames() {
        return $http.get(webdataUrl + 'resultSessions').then(function(response) {
            return response.data;
        });
    }

    function getMidraceResults() {
        return $http.get(webdataUrl + 'midraceResults').then(function(response) {
            return response.data;
        });
    }

    function saveMidraceResults(standings) {
        var urlPart = 'midraceResults';
        postOverlayDataToClients(urlPart, standings);

        return $http.post(webdataUrl + urlPart, standings);
    }

    function saveResultSessionNames(sessionNames) {
        var urlPart = 'resultSessions';
        postOverlayDataToClients(urlPart, sessionNames);

        return $http.post(webdataUrl + urlPart, sessionNames);
    }

    function getStartingOrder() {
        return $http.get(webdataUrl + 'startingOrder').then(function(response) {
            return response.data;
        });
    }

    function saveStartingOrder(standings) {
        var urlPart = 'startingOrder';
        postOverlayDataToClients(urlPart, standings);

        return $http.post(webdataUrl + urlPart, standings);
    }

    function getReplayEvents() {
        return $http.get(webdataUrl + 'replayEvents').then(function(response) {
            return response.data;
        });
    }

    function setReplayEvents(replayEvents) {
        var urlPart = 'replayEvents';
        postOverlayDataToClients(urlPart, replayEvents);

        return $http.post(webdataUrl + urlPart, replayEvents);
    }

    function getTrackMap () {
        return $http.get(restUrl + 'trackmap');
    }

    function jumpToReplayTime(time) {
        var urlPart = 'replayTime/' + time;
        sendPutRequestToClients(urlPart);

        return $http.put(restUrl + urlPart, null);
    }

    function focusOnDriver(slotID) {
        var urlPart = 'focus/' + slotID;
        sendPutRequestToClients(urlPart);

        return $http.put(restUrl + urlPart, null);
    }

    function getCameras() {
        return [
            {
                type: 'SCV_COCKPIT',
                name: 'On-board',
                index: 0,
                shouldAdvance: true
            },
            {
                type: 'SCV_COCKPIT',
                name: 'Cockpit',
                index: 1,
                shouldAdvance: false
            },
            {
                type: 'SCV_NOSECAM',
                name: 'Nose',
                index: 2,
                shouldAdvance: false
            },
            {
                type: 'SCV_SWINGMAN',
                name: 'Swingman',
                index: 3,
                shouldAdvance: true
            },
            {
                type: 'SCV_TRACKSIDE',
                name: 'Trackside',
                index: 4,
                shouldAdvance: true
            },
            {
                type: 'SCV_SPECTATOR',
                name: 'Spectator',
                index: 5,
                shouldAdvance: true
            }
        ]
    }

    function getActiveCamera() {
        return $http.get(restUrl + 'activeCamera').then(function(response) {
            return response.data;
        });
    }

    function selectCamera(type, trackSideGroup, shouldAdvance) {
        var urlPart = 'focus/' + type + '/' + trackSideGroup + '/' + shouldAdvance;
        sendPutRequestToClients(urlPart);

        return $http.put(restUrl + urlPart, null);
    }

    function sendPutRequestToClients(urlPart, data = null) {
        if (clients.length > 0) {
            _.forEach(clients, function(client) {
                var clientUrl = location.protocol + '//' + client.ip + ':' + client.port + restUrl + urlPart;

                $http.put(clientUrl, data);
            });
        }
    }

    function getDisplayOptions() {
        return $http.get(optionsRestUrl + 'display').then(function(response) {
            return response.data;
        });
    }

    function getSeasonStandingsJson(customConfigName) {
        return $http.get('customize/' + customConfigName + '/standings.json').then(function(response) {
            return response.data;
        });
    }

    function getScheduleJson(customConfigName) {
        return $http.get('customize/' + customConfigName + '/schedule.json').then(function(response) {
            return response.data;
        });
    }

    function getCarClassGroups(sortedOverallStandings) {
        return _.groupBy(sortedOverallStandings, 'carClass');
    }

    function addStandingsProps(standings, carClassGroups) {
        // Add class position
        var sortedOverallStandings = _.sortBy(standings, 'position');

        _.forOwn(carClassGroups, function (entries) {
            var classPosition = 1;

            _.forEach(entries, function (entry) {
                entry.classPosition = classPosition;
                classPosition++;
            });
        });

        addClassLeaderGapProps(sortedOverallStandings);
        addClassGapProps(sortedOverallStandings);
    }

    function addClassLeaderGapProps(sortedOverallStandings) {
        _.forEach(sortedOverallStandings, function (entry, index) {
            entry.timeBehindLeaderInClass = 0;
            entry.lapsBehindLeaderInClass = 0;
            if (entry.classPosition === 1) {
                return;
            }

            var leaderInClassOverallIndex = _.findIndex(
                sortedOverallStandings,
                { carClass: entry.carClass, classPosition: 1 }
            );

            if (leaderInClassOverallIndex === -1) {
                return;
            }

            entry.lapsBehindLeaderInClass = standingsService.getLapDifference(
                sortedOverallStandings[leaderInClassOverallIndex],
                entry
            );
            if (entry.lapsBehindLeaderInClass > 0) {
                entry.timeBehindLeaderInClass = 0;
            } else {
                var gap = 0;
                for (var i = leaderInClassOverallIndex + 1; i <= index; i++) {
                    gap += Math.abs(sortedOverallStandings[i].timeBehindNext);
                }
                entry.timeBehindLeaderInClass = gap;
            }
        });
    }

    function addClassGapProps(sortedOverallStandings) {
        _.forEach(sortedOverallStandings, function (entry, index) {
            // Add props for time behind car in front in class
            var timeBehindNextInClass;
            var lapsBehindNextInClass = 0;
            var nextInClassIndex = _.findLastIndex(
                sortedOverallStandings,
                { carClass: entry.carClass },
                index - 1
            );

            if (index === 0 || nextInClassIndex === -1) {
                // Overall leader or class leader
                timeBehindNextInClass = 0;
            } else {
                lapsBehindNextInClass = standingsService.getLapDifference(
                    sortedOverallStandings[nextInClassIndex],
                    entry
                );
                if (lapsBehindNextInClass > 0) {
                    timeBehindNextInClass = 0;
                } else {
                    var gap = 0;
                    for (var i = nextInClassIndex + 1; i <= index; i++) {
                        gap += Math.abs(sortedOverallStandings[i].timeBehindNext);
                    }

                    timeBehindNextInClass = gap;
                }
            }

            entry.lapsBehindNextInClass = lapsBehindNextInClass;
            entry.timeBehindNextInClass = timeBehindNextInClass;

            // Add props for time ahead of car behind in class
            var timeAheadPreviousInClass;
            var lapsAheadPreviousInClass;
            var previousInClassIndex = _.findIndex(
                sortedOverallStandings,
                { carClass: entry.carClass },
                index + 1
            );

            if (index === sortedOverallStandings.length - 1) {
                // Overall last
                timeAheadPreviousInClass = 0;
            } else {
                lapsAheadPreviousInClass = standingsService.getLapDifference(
                    entry,
                    sortedOverallStandings[previousInClassIndex]
                );
                if (lapsAheadPreviousInClass > 0) {
                    timeAheadPreviousInClass = 0;
                } else {
                    var gap = 0;
                    for (var i = index + 1; i <= previousInClassIndex; i++) {
                        lapsAheadPreviousInClass = sortedOverallStandings[i].lapsBehindNext;
                        gap += Math.abs(sortedOverallStandings[i].timeBehindNext);
                    }

                    timeAheadPreviousInClass = gap;
                }
            }

            entry.lapsAheadPreviousInClass = lapsAheadPreviousInClass;
            entry.timeAheadPreviousInClass = timeAheadPreviousInClass;
        });
    }

    function getCarClassColorFromConfig(customConfig, carClassName, propName = 'backgroundColor') {
        try {
            var color = customConfig.colors.carClasses[carClassName][propName];
            if (!color || !utilService.validateHexColor(color)) {
                return null;
            }

            return color;
        } catch(e) {
            return null;
        }
    }

    function getBorderStyle(carClass, customConfig, borderProp) {
        if (!carClass || !customConfig || !borderProp) {
            return null;
        }

        var classColor = getCarClassColorFromConfig(customConfig, carClass);

        if (!classColor ) {
            return null;
        }

        var styleObj = {};
        styleObj[borderProp] = 'solid 0.2em ' + classColor;

        return styleObj;
    }

    function getDriverBorderStyle(driver, customConfig, borderProp) {
        if (!driver || !customConfig || !borderProp) {
            return null;
        }

        return getBorderStyle(driver.carClass, customConfig, borderProp);
    }

    function getCarClassColorStyle(driver, customConfig, colorProp) {
        if (!driver || !customConfig || !colorProp) {
            return null;
        }

        var classColor = getCarClassColorFromConfig(
            customConfig,
            driver.carClass,
            colorProp
        );

        if (!classColor) {
            return null;
        }

        var styleObj = {};
        styleObj[colorProp] = classColor;

        return styleObj;
    }

    function getAllVehicles() {
        return $http.get(sessionsRestUrl + 'getAllVehicles').then(function(response) {
            return response.data;
        });
    }

    function getRaceResults() {
        return $http.post(sessionsRestUrl + 'getRaceResults').then(function(response) {
            return response.data;
        });
    }

    function getCarGameImageSrc(vehicleFilename) {
        if (!vehicleFilename) {
            return null;
        }

        return `/start/images/cars/${vehicleFilename}_frontAngle.webp`;
    }

    // Get car image from game
    function getCarGameImage(src) {
        if (!src) {
            return $q.reject('No image source provided');
        }

        return $http.get(src).then(function(image) {
            return image;
        });
    }

    function getCarImageSrc(entry, customConfigName) {
        if (!entry || !customConfigName) {
            return null;
        }

        var suffix = vehicleInfoService.getDriverOverrideField(entry, 'carImage.suffix');
        if (!suffix) {
            suffix = vehicleInfoService.getTeamOverrideField(entry, 'carImage.suffix') || '';
        }

        var carImagePath = 'customize/' + customConfigName + '/images/car/';
        var imageFileName = entry['vehicleFilename'] + suffix + '.png';
        var replaceInvalidFileNameCharsWith = '_';
        if (imageFileName) {
            imageFileName = imageFileName.replace(/[<>:"/\|?*]/g, replaceInvalidFileNameCharsWith);
        }

        return carImagePath + imageFileName;
    }

    function getTeamImageSrc(driver, customConfig) {
        var customConfigName = _.get(customConfig, 'configName');
        var teamName = vehicleInfoService.getTeamNameOverride(driver);
        var replaceInvalidFileNameCharsWith = '_';
        if (teamName) {
            teamName = teamName.replace(/[<>:"/\|?*]/g, replaceInvalidFileNameCharsWith);
        }

        if (customConfigName && teamName) {
            return 'customize/' + customConfigName + '/images/team/' + teamName + '.png';
        }
    }

    // Checks whether a car image is available as a file on disk, or as a fallback directly from the game. The src to
    // the available image is provided as a param in the callbackFn.
    function getDiskOrGameCarImageSrc(driver, customConfigName, callbackFn, vehicleFilename = null) {
        if (!driver || !customConfigName) {
            callbackFn(null, driver);
        }

        const diskCarImageSrc = getCarImageSrc(driver, customConfigName);

        function getAndCallbackWithCarGameImage() {
            if (!vehicleFilename) {
                vehicleFilename = driver.vehicleFilename;
            }

            const carGameImageSrc = getCarGameImageSrc(vehicleFilename);

            if (carGameImageSrc) {
                getCarGameImage(carGameImageSrc).then(function() {
                    callbackFn(carGameImageSrc, driver);
                }).catch(function() {
                    callbackFn(null, driver);
                });
            } else {
                callbackFn(null, driver);
            }
        }

        if (diskCarImageSrc) {
            $http.get(diskCarImageSrc).then(function() {
                callbackFn(diskCarImageSrc, driver);
            }).catch(function() {
                getAndCallbackWithCarGameImage();
            });
        } else {
            getAndCallbackWithCarGameImage();
        }
    }

    function displayGreenSector(driver, sector, sessionInfo) {
        if (!sessionService.isRaceSession(sessionInfo) && driver.pitting) {
            // Driver in the pits in a non-race session.
            // Compare driver's best lap's sector time to driver's fastest overall sector time.
            var bestLapSectorTime = standingsService.getDriverBestLapSectorTime(driver, sector);
            var bestSectorTime = standingsService.getDriverBestSectorTime(driver, sector);

            return bestLapSectorTime === bestSectorTime;
        }

        // In a race session or a non-race session and not pitting.
        // Compare last or previous sector time to driver's fastest overall sector time.
        if (sector === 1) {
            if (standingsService.isGreenSectorTime(driver, sector, 'current')) {
                return true;
            }
        } else if (sector === 2) {
            if (driver.currentSectorTime1 === -1) {
                // Driver is currently in sector 1
                if (standingsService.isGreenSectorTime(driver, sector, 'last')) {
                    return true;
                }
            } else {
                // Driver is currently in sector 2 or 3
                if (standingsService.isGreenSectorTime(driver, sector, 'current')) {
                    return true;
                }
            }
        } else if (sector === 3) {
            if (standingsService.isGreenSectorTime(driver, sector, 'last')) {
                return true;
            }
        }

        return false;
    }

    function getPurpleSectorTime(carClass, sector, sessionInfo) {
        var purpleSectorsInClass = _.get(sessionInfo, 'lapTimeInfo.purpleSectorsInClass');

        if (!purpleSectorsInClass || !sector || !sessionInfo) {
            return false;
        }

        var carClassPurpleSector = purpleSectorsInClass[carClass];

        return _.get(carClassPurpleSector, sector + '.time');
    }

    function displayPurpleSector(driver, sector, isRaceSession, purpleSectorTime) {
        if (driver.pitting && !isRaceSession) {
            return purpleSectorTime === standingsService.getDriverBestLapSectorTime(driver, sector, false);
        } else {
            if (sector === 1) {
                if (driver.currentSectorTime1 === purpleSectorTime) {
                    return true;
                }
            } else if (sector === 2) {
                if (driver.currentSectorTime1 === -1) {
                    // Driver is currently in sector 1
                    if (driver.lastSectorTime2 - driver.lastSectorTime1 === purpleSectorTime) {
                        return true;
                    }
                } else {
                    if (driver.currentSectorTime2 - driver.currentSectorTime1 === purpleSectorTime) {
                        return true;
                    }
                }
            } else if (sector === 3) {
                if (driver.lastLapTime - driver.lastSectorTime2 === purpleSectorTime) {
                    return true;
                }
            }
        }

        return false;
    }

    function disableOtherOverlays(currentOverlay, overlays) {
        if (!currentOverlay.disableOverlays || currentOverlay.disableOverlays.length === 0) {
            return false;
        }

        var disabledAnotherOverlay = false;

        /*
        Check and disable overlays that are set to disable all other overlays. This is for convenience and to reduce
        having to explicitly configure which overlays get disabled. For example the video overlay disables all other
        overlays when it's enabled. But you also want to do the opposite when the video overlay is active, and you
        toggle on some other overlay.
        */
        _.forEach(overlays, function(overlay) {
            if ((!overlay || !currentOverlay) || overlay.key === currentOverlay.key || overlay.disallowAutomaticDisabling) {
                return; // continue
            }

            if (overlay.disableOverlays && overlay.disableOverlays.group === 'all') {
                if (!_.includes(overlay.disableOverlays.except, currentOverlay.key)) {
                    overlay.visible = false;
                    disabledAnotherOverlay = true;
                }
            }
        });

        if (!currentOverlay.disableOverlays) {
            return disabledAnotherOverlay;
        }

        if (currentOverlay.disableOverlays.group && currentOverlay.disableOverlays.group === 'all') {
            _.forOwn(overlays, function (otherOverlay) {
                if (!otherOverlay || otherOverlay.visible === undefined || otherOverlay.key === currentOverlay.key
                    || otherOverlay.disallowAutomaticDisabling
                ) {
                    return;
                }

                var exceptList = currentOverlay.disableOverlays.except;

                if (exceptList && exceptList.length > 0) {
                    if (!_.includes(exceptList, otherOverlay.key)) {
                        otherOverlay.visible = false;
                        disabledAnotherOverlay = true;
                    }
                } else {
                    otherOverlay.visible = false;
                    disabledAnotherOverlay = true;
                }
            });
        } else {
            _.forEach(currentOverlay.disableOverlays, function (disableOverlay) {
                if (disableOverlay.disallowAutomaticDisabling !== true && overlays[disableOverlay].visible === true) {
                    overlays[disableOverlay].visible = false;
                    disabledAnotherOverlay = true;
                }
            });
        }

        return disabledAnotherOverlay;
    }

    function enableOtherOverlays(currentOverlay, overlays) {
        if (!currentOverlay.enableOverlays) {
            return false;
        }

        var enabledAnotherOverlay = false;

        _.forEach(currentOverlay.enableOverlays.overlays, function(enableOverlayKey) {
            if (overlays[enableOverlayKey].visible === false) {
                overlays[enableOverlayKey].visible = true;
                enabledAnotherOverlay = true;
            }
        });

        return enabledAnotherOverlay;
    }

    // Returns true when all drivers have finished the session either normally or with DNF/DQ
    function allDriversHaveFinished(standings) {
        for (var i = 0; i < standings.length; i++) {
            if (!standingsService.driverHasFinishedSession(standings[i])) {
                return false;
            }
        }

        return true;
    }

    // Check (and update when needed) session results in control panel
    function checkSessionResults(sessionInfo, standings) {
        const currentSessionName = sessionInfo.session;
        const getAndSaveResultSessionNames = () => {
            let resultSessionNames;

            getResultSessionNames().then(function(resultSessions) {
                if (!_.includes(resultSessions, currentSessionName)) {
                    if (!_.isNil(currentSessionName)) {
                        resultSessions.push(currentSessionName);
                    }
                }

                resultSessionNames = resultSessions;
            }).catch(function() {
                resultSessionNames = [];
            }).finally(function() {
                saveResultSessionNames(resultSessionNames);
            });
        }

        if (currentSessionName.includes('RACE')) {
            if (sessionInfo.gamePhase !== 8) {
                return; // We don't have race results until gamePhase is 8 (CHECKERED).
            }

            if (raceResults) {
                getAndSaveResultSessionNames();
            } else {
                getRaceResults().then((results) => {
                    raceResults = results.RaceResults.Race.Driver;

                    if (raceResults) {
                        saveSessionResults(mapRaceResultsToStandings(raceResults), currentSessionName);
                        getAndSaveResultSessionNames();
                    }
                });
            }
        } else {
            getSessionResults(currentSessionName).then(function(response) {
                var currentSessionResults; // Results for the current running session
                var newSessionResults;

                if (response.status === 200) {
                    currentSessionResults = response.data;
                }

                if (!currentSessionResults) {
                    currentSessionResults = [];
                }

                if (sessionService.sessionIsFinishing(sessionInfo)) {
                    newSessionResults = updateSessionResults(currentSessionResults, standings, sessionInfo);
                } else {
                    if (allDriversHaveFinished(standings) || sessionService.sessionHasRedFlag(sessionInfo)) {
                        newSessionResults = currentSessionResults;
                    } else {
                        if (sessionService.isBeforeSession(sessionInfo)) {
                            newSessionResults = [];
                        } else if (currentSessionResults && currentSessionResults.length > 0
                          && sessionService.sessionHasGreenFlag(sessionInfo)) {
                            newSessionResults = [];
                        }
                    }
                }

                if (newSessionResults) {
                    saveSessionResults(newSessionResults, currentSessionName);
                }

                getAndSaveResultSessionNames();
            });
        }
    }

    // Map the race results from the results XML format to the format we've been using with SessionResultsOverlay.
    function mapRaceResultsToStandings(raceResults) {
        return raceResults.map((result) => {
            const vehExtensionIndex = result.VehFile.lastIndexOf('.');

            return {
                bestLapTime: result.BestLapTime ? parseFloat(result.BestLapTime) : -1,
                carNumber: result.CarNumber,
                carClass: result.CarClass,
                classPosition: parseFloat(result.ClassPosition),
                driverName: result.Name,
                finishStatus: finishStatusFromResult(result.FinishStatus),
                finishTime: parseFloat(result.FinishTime),
                fullTeamName: result.TeamName,
                lapsCompleted: parseFloat(result.Laps),
                position: parseFloat(result.Position),
                vehicleFilename: result.VehFile.slice(0, vehExtensionIndex),
            }
        });
    }

    function finishStatusFromResult(resultFinishStatus) {
        switch (resultFinishStatus) {
            case 'None':
                return 'FSTAT_NONE';
            case 'Finished Normally':
                return 'FSTAT_FINISHED';
            case 'DNF':
                return 'FSTAT_DNF';
            case 'DQ':
                return 'FSTAT_DQ';
            default:
                return 'FSTAT_NONE';
        }
    }

    function allBestLapTimesAreInvalid(standings) {
        for (var i = 0; i < standings.length; i++) {
            if (standings[i].bestLapTime > -1) {
                return false;
            }
        }

        return true;
    }

    function updateSessionResults(currentSessionResults, standings, sessionInfo) {
        if (allBestLapTimesAreInvalid(standings)) {
            // Do not update results when all drivers' best lap times are invalid. Seems like this can happen when
            // session is switching and it would mess up the results.
            return currentSessionResults;
        }

        _.forEach(standings, function(currentStandingsEntry) {
            var currentSessionResultsIndex = _.findIndex(
                currentSessionResults,
                { slotID: currentStandingsEntry.slotID }
            );

            if (currentSessionResultsIndex === -1) {
                // Driver not found in results, add driver
                if (sessionService.isRaceSession(sessionInfo)) {
                    if (standingsService.driverHasFinishedSession(currentStandingsEntry)) {
                        // In race session add driver to results as they finish
                        currentSessionResults.push(currentStandingsEntry);
                    }
                } else {
                    currentSessionResults.push(currentStandingsEntry);
                }
            } else {
                // Driver already in results, update their result
                if (sessionService.isRaceSession(sessionInfo)) {
                    if (!standingsService.driverHasFinishedSession(currentSessionResults[currentSessionResultsIndex])) {
                        currentSessionResults[currentSessionResultsIndex] = currentStandingsEntry;
                    }
                } else {
                    currentSessionResults[currentSessionResultsIndex] = currentStandingsEntry;
                }
            }
        });

        return currentSessionResults;
    }

    function getCustomConfigSeriesName(customConfig) {
        return _.get(customConfig, 'seriesName');
    }

    function getCustomConfigTrackName(customConfig) {
        return _.get(customConfig, 'trackName');
    }

    function getCustomConfigSeriesColor(customConfig, getTextColor = false) {
        var colorProp = getTextColor ? 'color' : 'backgroundColor';
        var seriesColor = _.get(customConfig, 'colors.series.' + colorProp);

        if (!seriesColor || !utilService.validateHexColor(seriesColor)) {
            seriesColor = getTextColor ? '#fff' : '#db0618';
        }

        return seriesColor;
    }

    function getDefaultSeriesLogoUrl() {
        return '/framework/images/rfactor_2_racing_series_narrow.png';
    }

    function getGameLogoUrl() {
        return '/framework/images/rf2_logo.png';
    }

    function isMixedClassMode(selectedCarClass) {
        return selectedCarClass === 'Mixed';
    }

    function isMulticlassClassMode(selectedCarClass) {
        return selectedCarClass === 'Multiclass';
    }

    function getCurrentRoundIndex(schedule) {
        var currentDate = new Date();
        currentDate.setHours(0, 0, 0, 0);

        var scheduleIndex = _.findIndex(schedule, function(entry) {
            var entryDate = new Date(entry.date);
            entryDate.setHours(0, 0, 0, 0);

            return currentDate.getTime() === entryDate.getTime();
        });

        return scheduleIndex;
    }

    function getCurrentTrackNameFromSchedule(schedule, customConfig = null, sessionInfo = null) {
        var trackName;
        var currentRoundIndex = getCurrentRoundIndex(schedule);
        if (currentRoundIndex > -1) {
            trackName = _.get(schedule, currentRoundIndex + '.trackName');
        }

        if (!trackName && customConfig) {
            trackName = getCustomConfigTrackName(customConfig);
        }

        if (!trackName && sessionInfo) {
            trackName = _.get(sessionInfo, 'trackName');
        }

        return trackName;
    }

    function getAvailableDynamicVehicleClassColors() {
        return [
            {
                background: '#fff',
                text: '#000'
            },
            {
                background: '#ff8000',
                text: '#000'
            },
            {
                background: '#ffea00',
                text: '#000'
            },
            {
                background: '#00ffff',
                text: '#000'
            },
            {
                background: '#aaff00',
                text: '#000'
            },
            {
                background: '#4400cc',
                text: '#fff'
            },
            {
                background: '#26404d',
                text: '#fff'
            },
            {
                background: '#4d3636',
                text: '#fff'
            },
            {
                background: '#4d4d4d',
                text: '#fff'
            },
            {
                background: '#000',
                text: '#fff'
            }
        ];
    }

    function getDynamicVehicleClassColors(vehicleClasses) {
        let dynamicVehicleClassColors = [];
        vehicleClasses = _.orderBy(vehicleClasses, null); // Assign colors alphabetically.

        _.forEach(vehicleClasses, (vehicleClass, index) => {
            const dynamicClassColor = this.getAvailableDynamicVehicleClassColors()[index];

            if (dynamicClassColor) {
                dynamicVehicleClassColors.push({
                    vehicleClass: vehicleClass,
                    background: dynamicClassColor.background,
                    text: dynamicClassColor.text
                });
            }
        });

        return dynamicVehicleClassColors;
    }

    function findVehicleClassesNeedingDynamicColor(standings, vehicleClassesWithConfiguredColors) {
        if (!standings || !vehicleClassesWithConfiguredColors) {
            return [];
        }

        let vehicleClasses = _.uniq(_.map(standings, 'carClass'));

        // Filter out vehicle classes that have explicitly defined color.
        vehicleClasses = _.filter(
            vehicleClasses,
            (vehicleClass) => !_.find(vehicleClassesWithConfiguredColors, {
                vehicleClass: vehicleClass
            })
        );

        return vehicleClasses;
    }

    function getVehicleClassesInfo() {
        return $http.get('/framework/json/vehicle_classes.json').then(response => response.data);
    }
});
