angular.module('SessionService', ['UtilService']).service('sessionService', function (utilService) {
    var service = {
        isQualifySession: isQualifySession,
        isRaceSession: isRaceSession,
        sessionIsInvalid: sessionIsInvalid,
        getSessionType: getSessionType,
        isTimedSession: isTimedSession,
        isLapsAndTimeRaceSession: isLapsAndTimeRaceSession,
        displaySessionName: displaySessionName,
        displaySessionDuration: displaySessionDuration,
        getYellowSector: getYellowSector,
        isBeforeSession: isBeforeSession,
        sessionHasGreenFlag: sessionHasGreenFlag,
        sessionHasRedFlag: sessionHasRedFlag,
        sessionIsFinishing: sessionIsFinishing,
        getRaceLapsRemaining: getRaceLapsRemaining,
        getTimeRemaining: getTimeRemaining
    }

    return service;

    function isQualifySession(sessionInfo) {
        return sessionInfo && sessionInfo.session && sessionInfo.session.indexOf('QUALIFY') !== -1;
    }

    function isRaceSession (sessionInfo) {
        return sessionInfo && sessionInfo.session && sessionInfo.session.indexOf('RACE') !== -1;
    }

    function sessionIsInvalid(sessionInfo) {
        return sessionInfo && sessionInfo.session === 'INVALID';
    }

    function getSessionType(sessionName) {
        if (!sessionName) {
            return null;
        }

        return sessionName.replace(/\d/g, '');
    }

    function isTimedSession(sessionInfo) {
        return sessionInfo && sessionInfo.maximumLaps > 1500;
    }

    function isLapsAndTimeRaceSession(sessionInfo) {
        return sessionInfo && sessionInfo.raceCompletion
            && sessionInfo.raceCompletion.timeCompletion !== undefined
            && sessionInfo.raceCompletion.lapsCompletion !== undefined
            && isRaceSession(sessionInfo)
    }

    function displaySessionName(sessionInfo) {
        if (!sessionInfo) {
            return '-';
        }

        var sessionName = getSessionType(sessionInfo.session);

        if (!sessionName) {
            return '-';
        }

        return getSessionType(sessionInfo.session).toLowerCase();
    }

    function displaySessionDuration(sessionInfo, standings) {
        var durationText = '-';

        if (!sessionInfo) {
            return durationText;
        }

        if (!isRaceSession(sessionInfo)) {
            durationText = displaySessionTimeLeft(sessionInfo);
        } else if (sessionInfo.maximumLaps !== 2147483647) {
            var lapsCompleted = getTotalLapsCompletedInRace(standings);

            durationText = "LAP " + (lapsCompleted + 1) + "/" + sessionInfo.maximumLaps;

			if (lapsCompleted == sessionInfo.maximumLaps) {
                durationText = "LAST LAP";
            }
		} else if (sessionInfo.endEventTime > 0) {
            // Timed race
            durationText = displaySessionTimeLeft(sessionInfo);
        }

        return durationText;
    }

    function getTotalLapsCompletedInRace(standings) {
        var leader = _.find(standings, { position: 1 });

        if (!leader) {
            return -1;
        }

        return leader.lapsCompleted;
    }

    function displaySessionTimeLeft (sessionInfo) {
        var timeLeft = utilService.secToString(sessionInfo.endEventTime - sessionInfo.currentEventTime);

        if (timeLeft < 10) {
            timeLeft = '0:0' + timeLeft;
        } else if (timeLeft < 60) {
            timeLeft = '0:' + timeLeft;
        } else if (timeLeft === '-') {
            timeLeft = '0:00';
        }

        return timeLeft;
    }

    // Returns sector number if a sector is yellow, or 0 if no yellow sector
    function getYellowSector(sessionInfo) {
        if (!sessionInfo || !sessionInfo.sectorFlag) {
            return 0;
        }

        for (var i = 0; i < sessionInfo.sectorFlag.length; i++) {
            if (sessionInfo.sectorFlag[i] === 'YELLOW') {
                return i + 1;
            }
        }

        return 0;
    }

    function isBeforeSession(sessionInfo) {
        return sessionInfo && sessionInfo.gamePhase === 0;
    }

    function sessionHasGreenFlag(sessionInfo) {
        return sessionInfo && sessionInfo.gamePhase === 5;
    }

    function sessionHasRedFlag(sessionInfo) {
        return sessionInfo && sessionInfo.gamePhase === 7;
    }

    function sessionIsFinishing(sessionInfo) {
        return sessionInfo && sessionInfo.gamePhase === 8;
    }

    function getRaceLapsRemaining(sessionInfo, leader) {
        if (!sessionInfo || !leader || !isRaceSession(sessionInfo) || isTimedSession(sessionInfo)) {
            return -1;
        }

        return sessionInfo.maximumLaps - leader.lapsCompleted;
    }

    function getTimeRemaining(sessionInfo) {
        if (!sessionInfo) {
            return -1;
        }

        return sessionInfo.endEventTime - sessionInfo.currentEventTime;
    }
});
