angular.module(
    'StandingsService',
    ['SessionService', 'UtilService']
).service('standingsService', function (sessionService, utilService) {
    var service = {
        displayBestLapTimeDiffToLeader,
        displayBestLapTimeDiffToNextDriver,
        displayLastLapTimeDiffToNextDriver,
        displayLiveGapToLeader,
        displayLiveGapToNextDriver,
        getSessionFastestSectors,
        getSessionFastestSectorsInClass,
        findBestSectorTime,
        displayTime,
        getEntryFadedOut,
        findLeader,
        findNextDriver,
        findSessionBestLapTimeEntry,
        findSessionBestLapTime,
        getLapTimeDiff,
        formatDriverLapTime,
        formatLapTimeDiff,
        isDriverLapTimeValid,
        isLapTimeValid,
        getDriverBestLapSectorTime,
        getDriverBestSectorTime,
        getDriverSectorTime,
        timeToFixedPrecision,
        isGreenSectorTime,
        isDriverDnf,
        isDriverDq,
        driverHasFinishedSession,
        isDriverPitting,
        driverIsEnteringPitLane,
        driverIsInPitBox,
        driverIsExitingPitLane,
        driverIsInGarageStall,
        formatDriverStatus,
        getDriverTotalRaceTime,
        convertCarSpeed,
        getLapDifference,
        hasDrsActive,
        getTireCompoundName
    }

    return service;

    // Get driver one position ahead in overall standings
    function getNextDriver(driver, standings) {
        return _.find(standings, { position: driver.position - 1 });
    }

    // Get driver one position ahead in class standings
    function getNextDriverInClass(driver, standings) {
        return _.find(standings, { classPosition: driver.classPosition - 1, carClass: driver.carClass });
    }

    // Get difference in best lap time between leader and driver
    function getBestLapTimeDiffToLeader(driver, standings, inClass) {
        var leader = inClass ? findLeader(standings, driver.carClass) : findLeader(standings);

        if (!driver || !leader || driver.bestLapTime === -1 || leader.bestLapTime === -1) {
            return -1;
        }

        return driver.bestLapTime - leader.bestLapTime;
    }

    // Get difference in best lap time between the driver ahead and driver
    function getBestLapTimeDiffToNextDriver(driver, standings, inClass) {
        var nextDriver = inClass ? getNextDriverInClass(driver, standings) : getNextDriver(driver, standings);
        var posProp = inClass ? 'classPosition' : 'position';

        if (driver[posProp] === 1 && driver.bestLapTime !== -1) {
            return 0;
        }

        if (!driver || !nextDriver || driver.bestLapTime === -1 || nextDriver.bestLapTime === -1) {
            return -1;
        }

        return driver.bestLapTime - nextDriver.bestLapTime;
    }

    // Get difference in best lap time between the driver ahead and driver
    function getLastLapTimeDiffToNextDriver(driver, standings, inClass) {
        var nextDriver = inClass ? getNextDriverInClass(driver, standings) : getNextDriver(driver, standings);
        var posProp = inClass ? 'classPosition' : 'position';

        if (driver[posProp] === 1 && driver.lastLapTime !== -1) {
            return 0;
        }

        if (!driver || !nextDriver || driver.lastLapTime === -1 || nextDriver.lastLapTime === -1) {
            return -1;
        }

        return driver.lastLapTime - nextDriver.lastLapTime;
    }

    function displayBestLapTimeDiffToLeader(driver, standings, inClass, decimals = 3) {
        if (!driver || !standings) {
            return '-';
        }

        var timeDiff = displayTimeDiff(
            getBestLapTimeDiffToLeader(driver, standings, inClass),
            driver,
            inClass,
            'bestLapTime',
            decimals
        );

        if (timeDiff < 0) {
            return '-';
        }

        return timeDiff;
    }

    function displayBestLapTimeDiffToNextDriver(driver, standings, inClass, decimals = 3) {
        if (!driver || !standings) {
            return '';
        }

        var timeDiff = displayTimeDiff(
            getBestLapTimeDiffToNextDriver(driver, standings, inClass),
            driver,
            inClass,
            'bestLapTime',
            decimals
        );

        if (timeDiff < 0) {
            return '-';
        }

        return timeDiff;
    }

    function displayLastLapTimeDiffToNextDriver(driver, standings, inClass, decimals = 3) {
        if (!driver || !standings) {
            return '';
        }

        return displayTimeDiff(
            getLastLapTimeDiffToNextDriver(driver, standings, inClass),
            driver,
            inClass,
            'lastLapTime',
            decimals
        );
    }

    function displayTimeDiff(gap, driver, inClass, lapTimeProp, decimals = 3) {
        var posProp = inClass ? 'classPosition' : 'position';

        if (driver[posProp] === 1 && driver[lapTimeProp] !== -1) {
            return '';
        } else {
            if (gap > 0) {
                return '+' + gap.toFixed(decimals);
            }

            return gap.toFixed(decimals);
        }
    }

    function displayLiveGapToLeader(driver, inClass, decimals = 3) {
        if (!driver) {
            return '-';
        }

        var lapsBehindProp = inClass ? 'lapsBehindLeaderInClass' : 'lapsBehindLeader';
        var posProp = inClass ? 'classPosition' : 'position';
        var timeBehindProp = inClass ? 'timeBehindLeaderInClass' : 'timeBehindLeader';

        if (driver[lapsBehindProp] > 0) {
            return '+' + driver[lapsBehindProp] + 'L';
        }

        if (driver[posProp] === 1) {
            return '';
        }

        var gap = driver[timeBehindProp];

        if (!gap) {
            return '-';
        }

        gap = Math.abs(gap);

        return '+' + gap.toFixed(decimals);
    }

    function displayLiveGapToNextDriver(driver, inClass, decimals = 3) {
        if (!driver) {
            return '-';
        }

        var lapsBehindProp = inClass ? 'lapsBehindNextInClass' : 'lapsBehindNext';
        var posProp = inClass ? 'classPosition' : 'position';
        var timeBehindProp = inClass ? 'timeBehindNextInClass' : 'timeBehindNext';

        if (driver[lapsBehindProp] > 0) {
            return '+' + driver[lapsBehindProp] + 'L';
        }

        if (driver[posProp] === 1) {
            return '';
        }

        var gap = '';
        var timeBehind = driver[timeBehindProp];

        if (timeBehind === null || timeBehind === undefined) {
            return '';
        }

        timeBehind = Math.abs(timeBehind);
        gap += '+' + timeBehind.toFixed(decimals);

        return gap;
    }

    function findSessionBestLapTimeEntry(drivers, carClass) {
        if (carClass) {
            drivers = _.filter(drivers, { carClass: carClass });
        }

        var bestLapEntry = _.minBy(drivers, function(entry) {
            if (isDriverLapTimeValid(entry, 'bestLapTime')) {
                return entry.bestLapTime;
            }
        });

        if (!bestLapEntry) {
            return null;
        }

        return bestLapEntry;
    }

    function findSessionBestLapTime(drivers, carClass) {
        var bestLapTimeEntry = findSessionBestLapTimeEntry(drivers, carClass);

        if (!bestLapTimeEntry) {
            return -1;
        }

        return bestLapTimeEntry.bestLapTime;
    }

    function getSessionFastestSectors(history) {
        var purpleSectors = {};

        for (var sector = 1; sector <= 3; sector++) {
            var fastestSector = {
                time: Infinity,
                slotID: -1
            }

            _.forOwn(history, function(historyEntries) {
                _.forEach(historyEntries, function(carEntry) {
                    fastestSector = findFasterSectorTime(sector, carEntry, fastestSector);
                });
            });

            purpleSectors[sector] = fastestSector;
        }

        return purpleSectors;
    }

    function getSessionFastestSectorsInClass(history, carClasses) {
        var carClassPurpleSectors = {};

        _.forOwn(carClasses, function(value, key) {
            carClassPurpleSectors[key] = {};

            for (var sector = 1; sector <= 3; sector++) {
                var fastestSector = {
                    time: Infinity,
                    slotID: -1
                };

                _.forOwn(history, function(historyEntries) {
                    _.forEach(historyEntries, function(carEntry) {
                        if (!carEntry || carEntry.carClass !== key) {
                            return;
                        }

                        fastestSector = findFasterSectorTime(sector, carEntry, fastestSector);
                    });

                    carClassPurpleSectors[key][sector] = fastestSector;
                });
            }
        });

        return carClassPurpleSectors;
    }

    function findFasterSectorTime(sector, carEntry, fastestSector) {
        if (!carEntry) {
            return fastestSector;
        }

        if (sector === 1) {
            if (isDriverLapTimeValid(carEntry, 'sectorTime1') && carEntry.sectorTime1 < fastestSector.time) {
                fastestSector = {
                    time: carEntry.sectorTime1,
                    slotID: carEntry.slotID
                }
            }
        } else if (sector === 2) {
            if (isDriverLapTimeValid(carEntry, 'sectorTime1')
                && isDriverLapTimeValid(carEntry, 'sectorTime2')
                && (carEntry.sectorTime2 - carEntry.sectorTime1) < fastestSector.time) {
                fastestSector = {
                    time: carEntry.sectorTime2 - carEntry.sectorTime1,
                    slotID: carEntry.slotID
                }
            }
        } else {
            if (isDriverLapTimeValid(carEntry, 'lapTime')
                && isDriverLapTimeValid(carEntry, 'sectorTime2')
                && (carEntry.lapTime - carEntry.sectorTime2) < fastestSector.time) {
                fastestSector = {
                    time: carEntry.lapTime - carEntry.sectorTime2,
                    slotID: carEntry.slotID
                }
            }
        }

        return fastestSector;
    }

    function findBestSectorTime(entryHistory, sector) {
        if (sector < 1 || sector > 3) {
            return -1;
        }

        let bestSectorTime = Infinity;

        _.forEach(entryHistory, function(historyEntry) {
            if (!historyEntry) {
                return;
            }

            if (sector === 1) {
                if (lapTimeIsValid(historyEntry.sectorTime1)
                    && historyEntry.sectorTime1 < bestSectorTime) {
                    bestSectorTime = historyEntry.sectorTime1;
                }
            } else if (sector === 2) {
                if (lapTimeIsValid(historyEntry.sectorTime2)
                    && lapTimeIsValid(historyEntry.sectorTime1)
                    && (historyEntry.sectorTime2 - historyEntry.sectorTime1) < bestSectorTime) {
                    bestSectorTime = historyEntry.sectorTime2 - historyEntry.sectorTime1;
                }
            } else {
                if (lapTimeIsValid(historyEntry.sectorTime2)
                    && lapTimeIsValid(historyEntry.lapTime)
                    && (historyEntry.lapTime - historyEntry.sectorTime2) < bestSectorTime) {
                    bestSectorTime = historyEntry.lapTime - historyEntry.sectorTime2;
                }
            }
        });

        if (bestSectorTime === Infinity) {
            return -1;
        }

        return bestSectorTime;
    }

    function lapTimeIsValid(lapTime) {
        return lapTime > 0;
    }

    function displayTime (entry, sessionInfo) {
        return isRaceSession(sessionInfo)
            ? utilService.secToString(entry.lastLapTime, 3, true)
            : utilService.secToString(entry.bestLapTime,3, true);
    }

    function getEntryFadedOut(entry, sessionInfo) {
        if (sessionService.isRaceSession(sessionInfo)) {
            return isDriverDnf(entry) || isDriverDq(entry);
        } else {
            return isDriverPitting(entry) || isDriverDnf(entry) || isDriverDq(entry);
        }
    }

    function findLeader(standings, carClass = null) {
        if (carClass) {
            return _.find(standings, { classPosition: 1, carClass: carClass });
        }

        return _.find(standings, { position: 1 });
    }

    function findNextDriver(standings, driver, carClass = null) {
        if (carClass) {
            return _.find(standings, { classPosition: driver.classPosition - 1, carClass: carClass });
        }

        return _.find(standings, { position: driver.position - 1 });
    }

    function getLapTimeDiff(driver1, driver2, lapTimeProp) {
        if (!isDriverLapTimeValid(driver1, lapTimeProp) || !isDriverLapTimeValid(driver2, lapTimeProp)) {
            return null;
        }

        return driver2[lapTimeProp] - driver1[lapTimeProp];
    }

    function formatDriverLapTime(driver, lapTimeProp, digits = 3, invalidDisplay = '-') {
        if (!isDriverLapTimeValid(driver, lapTimeProp)) {
            return invalidDisplay;
        }

        return formatLapTime(driver[lapTimeProp], digits);
    }

    function formatLapTime(lapTime, digits) {
        return utilService.secToString(lapTime, digits);
    }

    function formatLapTimeDiff(lapTimeDiff, digits = 3, invalidDisplay = '-') {
        if (lapTimeDiff === null || lapTimeDiff === undefined) {
            return invalidDisplay;
        }

        if (lapTimeDiff >= 0) {
            return '+' + lapTimeDiff.toFixed(digits);
        }

        return lapTimeDiff.toFixed(digits);
    }

    function isDriverLapTimeValid(driver, lapTimeProp) {
        return driver && lapTimeProp && isLapTimeValid(driver[lapTimeProp]);
    }

    function isLapTimeValid(lapTime) {
        return lapTime > 0;
    }

    // Returns the sector time for the best lap of the driver (not the best overall sector time for the driver!)
    function getDriverBestLapSectorTime(driver, sectorNumber, toFixed = true) {
        if (!driver || !sectorNumber || sectorNumber > 3) {
            return -1;
        }

        var bestLapSectorTime;

        if (sectorNumber === 2) {
            bestLapSectorTime = driver.bestLapSectorTime2 - driver.bestLapSectorTime1;
        } else if (sectorNumber === 3) {
            bestLapSectorTime = driver.bestLapTime - driver.bestLapSectorTime2;
        } else {
            bestLapSectorTime = driver['bestLapSectorTime' + sectorNumber];
        }

        if (toFixed) {
            // Fixed precision to make time comparison reliable.
            return timeToFixedPrecision(bestLapSectorTime);
        }

        return bestLapSectorTime;
    }

    // Returns the overall best sector time for the driver.
    function getDriverBestSectorTime(driver, sectorNumber, toFixed = true) {
        if (!driver || !driver.lapTimeInfo || !sectorNumber || sectorNumber > 3) {
            return -1;
        }

        if (toFixed) {
            // Get best sector from lapTimeInfo. Fixed precision to make time comparison reliable.
            return timeToFixedPrecision(driver.lapTimeInfo['bestSector' + sectorNumber]);
        }

        return driver.lapTimeInfo['bestSector' + sectorNumber];
    }

    // Get current or last sector time
    function getDriverSectorTime(driver, sectorNumber, sectorTimePropPart, toFixed = true) {
        if (!driver || !sectorNumber || sectorNumber > 3) {
            return -1;
        }

        var sectorTime;

        if (sectorNumber === 1) {
            sectorTime = driver[sectorTimePropPart + 'SectorTime' + sectorNumber];
        } else if (sectorNumber === 2) {
            sectorTime = driver[sectorTimePropPart + 'SectorTime2'] - driver[sectorTimePropPart + 'SectorTime1']
        } else if (sectorNumber === 3) {
            var lapTimePropName = sectorTimePropPart === 'bestLap' ? 'bestLapTime' : sectorTimePropPart + 'LapTime';
            sectorTime = driver[lapTimePropName] - driver[sectorTimePropPart + 'SectorTime2'];
        }

        if (!sectorTime || sectorTime < 0) {
            return -1;
        }

        if (toFixed) {
            return timeToFixedPrecision(sectorTime);
        }

        return sectorTime;
    }

    function timeToFixedPrecision(time, digits = 3) {
        if (time === null || time === undefined) {
            return null;
        }

        // Unary plus operator to convert back to number.
        return +time.toFixed(digits);
    }

    function isGreenSectorTime(driver, sector, lapTimePropPart) {
        if (!driver || driver[lapTimePropPart + 'LapTime'] === -1) {
            return false;
        }

        return getDriverSectorTime(driver, sector, lapTimePropPart) === getDriverBestSectorTime(driver, sector);
    }

    function isDriverDnf(driver) {
        // Support finishStatus being a number for now
        return driver && (driver.finishStatus === 'FSTAT_DNF' || driver.finishStatus === 2);
    }

    function isDriverDq(driver) {
        // Support finishStatus being a number for now
        return driver && (driver.finishStatus === 'FSTAT_DQ' || driver.finishStatus === 3);
    }

    // Returns true if driver has finished session normally or DQ/DNF
    function driverHasFinishedSession(driver) {
        // Support finishStatus being a number for now
        return driver && driver.finishStatus !== 'FSTAT_NONE' && driver.finishStatus !== 0;
    }

    function isDriverPitting(driver) {
        return driver && driver.pitting;
    }

    function driverIsEnteringPitLane(driver) {
        return driver && driver.pitState === 'ENTERING';
    }

    function driverIsInPitBox(driver) {
        return driver && driver.pitState === 'STOPPED';
    }

    function driverIsExitingPitLane(driver) {
        return driver && driver.pitState === 'EXITING';
    }

    function driverIsInGarageStall(driver) {
        return driver && driver.inGarageStall;
    }

    /**
     * @param {Object} driver
     * @param {DetailedPitState} detailedPitState
     * @returns {string}
     */
    function formatDriverStatus(driver, detailedPitState) {
        if (isDriverDnf(driver)) {
            return 'DNF';
        }

        if (isDriverDq(driver)) {
            return 'DQ';
        }

        if (detailedPitState) {
            return detailedPitState;
        }

        if (isDriverPitting(driver)) {
            return 'Pit';
        }

        return '';
    }

    function getDriverTotalRaceTime(driverHistory) {
        var totalRaceTime = 0;

        _.forEach(driverHistory, function(historyEntry) {
            if (historyEntry.lapTime > -1) {
                totalRaceTime += historyEntry.lapTime;
            }
        });

        return totalRaceTime;
    }

    function convertCarSpeed(speed, unit) {
        if (speed === undefined || speed === null) {
            return 0;
        }

        if (unit === 'mph') {
            return (speed * 2.2369);
        }

        return (speed * 3.6);
    }

    function getLapDifference(driverA, driverB) {
        if (!driverA || !driverB) {
            return null;
        }

        var lapDiff = driverA.lapsCompleted - driverB.lapsCompleted;
        if (lapDiff > 0) {
            if (driverA.lapDistance < driverB.lapDistance) {
                lapDiff--;
            }
        } else if (lapDiff < 0) {
            if (driverA.lapDistance > driverB.lapDistance) {
                lapDiff--;
            }
        }

        return lapDiff;
    }

    function hasDrsActive(entry) {
        return entry && entry.drsActive;
    }

    function getTireCompoundName(entry, axle) {
        if (!entry) {
            return null;
        }

        return entry['tireCompoundName' + axle];
    }
});
